#pragma once
////////////////////////////////////////////////////////////////
//
// BlanketModule - A class for Blanker module writers
//
// This is a lean, mean, blanking machine!  Designed to do
// the practical minimum to abstract out the notions
// of module, configure and saver.
//
// Construction and destruction of one of these corresponds with
// "module_initialize" and "module_cleanup"
//
////////////////////////////////////////////////////////////////

class BlanketModule : public BArchivable
{
	// Base class for a single Saver instance, the construction
	// and destruction of one of these corresponds with
	// "module_start_saving" and "module_end_saving"
	//
	// This class handles all the really grobbely stuff like
	// the threading, startup and termination. Does what is
	// required to avoid lockups through use of a semaphore.
	//
	// All you need to do to make use of this is to derive your saver
	// from this and provide an implementation for "Tick", in there
	// you draw into the View().
	//
	// Remember you will have to provide a derivation from
	// BlanketModule where you implement InstantiateSaver to
	// create the appropriate saver.
	class Saver
	{
	public:
		// The optional snoozetime argument indicates the Tick distance.
		// You can change the timing later using the SetSnoozeTime method.
		Saver(BlanketModule *owner, BView* view, bigtime_t snoozetime = 20000);
		virtual ~Saver();

		// Start the Ticks
		// This method has to be called after construction, to avoid race
		// conditions. It has to be called by the caller of
		// BlanketModule::InstantiateSaver.
		// The Tick thread is always created at B_LOW_PRIORITY, although
		// it's easy to change that, you are strongly encouraged to
		// leave it alone.
		// You can call Start if you previously called a Stop.
		void Start();

		// Stop the Ticks
		// It has to be called by the caller of
		// BlanketModule::InstantiateSaver before destruction
		// You can call Stop if you want to stop ticking.
		void Stop();

		// Allow snooze time to be set dynamically,
		// this is the period between ticks!
		void SetSnoozeTime(bigtime_t st) { m_snoozetime = st; }

		// This virtual is called on initialization,
		// when the screen has to be set up.
		// It rarely needs to be overridden, it's currently
		// used by the DesktopSaver class, check that for details.
		virtual void Initialize();

		// This virtual is called on every clock tick
		// You should implement this one and do your drawing
		virtual void Tick();

		// A few simple read-only accessors
		BView* View() const { return m_view; }
		bigtime_t SnoozeTime() const { return m_snoozetime; }

	private:
		// Main object based thread entry point
		long ThreadEntry();

		static long thread_entry_stub(void*);

		BView* m_view;
		thread_id m_thread;
		sem_id m_sem;
		bigtime_t m_snoozetime;
	};

	// Base class for a single module configuration object
	// the construction and destruction of one of these corresponds
	// with "module_start_config" and "module_end_config"
	//
	// When the constructor is called you are guaranteed that the view
	// is valid and attached to a window.
	class Config
	{
	public:
		// The wantpreview flag is used to let the Config constructor
		// automatically build a preview view, that can be later
		// retrieved through GetPreview. Although not extremely
		// elegant, it hides some tricky details that might cause
		// deadlocks (like calling the Saver::Stop method in the
		// window thread); the focus of this framework is on
		// simplicity, afterall.
		//
		// If wantpreview is true, the constructor will instantiate a
		// Saver through BlanketModule::InstantiateSaver, which will be
		// called with preview=true.
		Config(BlanketModule *owner, BView* view, bool wantpreview);
		virtual ~Config();

		// This handy method will return a complete preview
		// view for your module, it's your responsibility
		// to attach it to your configuration view (or
		// otherwise make sure it's deleted).
		BView *GetPreview(BRect frame, uint32 resizeMask, uint32 flags);

		// A few simple read-only accessors
		BView *View() const { return c_view; }
		BlanketModule *Owner() const { return c_owner; }

	private:
		BView *c_view;
		BlanketModule *c_owner;
		BlanketModule::Saver *c_saver;
		BView *c_preview;
	};

public:
	// You'll get your settings back in the constructor BMessage
	// settings will be saved through the Archive method, if you
	// implement it. Setting management is handled through the
	// BArchivable interface that BlanketModule derives from.
	BlanketModule(BMessage* archive);
	virtual ~BlanketModule();

	// Standard stuff for a BArchivable... you better
	// implement this baby!
	static BArchivable *Instantiate(BMessage*);

	// Blanket uses these virtuals to coordinate activity...
	// The Saver instantiation method accepts a preview flag,
	// that might be useful for BWindowScreen based
	// modules or modules that muck the desktop.
	virtual Saver* InstantiateSaver(BView* view, bool preview);
	virtual Config* InstantiateConfig(BView* view);
};
