#include <ScrollView.h>
#include <TextView.h>
#include <StorageKit.h>
#include <E-mail.h>

#if __INTEL__
	#include <add-ons/tracker/TrackerAddOn.h>
#else
	#pragma export on
	void process_refs(entry_ref dir_ref, BMessage *msg, void *);
	#pragma export reset
#endif

#include <ctype.h>
#include <parsedate.h>
#include <stdio.h>
#include <string.h>
#include <time.h>

#define NOT_MAIL 'notm'

static const char FIELD_TO[] = "to: ";
static const char FIELD_REPLY[] = "reply-to: ";
static const char FIELD_SUBJECT[] = "subject: ";
static const char FIELD_WHEN[] = "date: ";
static const char FIELD_PRIORITY[] = "priority: ";
static const char FIELD_FROM[] = "from: ";
static const char FIELD_MIME[] = "mime-version: ";

template<class T>
class AutoDel // auto-by-del?
{
	T *ptr;
public:
	AutoDel(T *p) { ptr = p; };
	~AutoDel() { delete ptr; };
};

status_t fix_mail_attr(BFile *file)
{
	status_t err;
	int32 cur, end;
	off_t message_size;
	char *message;
	char separator = 0;
	bool is_mail = false;

	if ((err = file->GetSize(&message_size)) != B_NO_ERROR) {
		return err;
	}

	if (message_size == 0) {
		return B_NO_ERROR;
	}

	message = new char[message_size];
	AutoDel<char> autodel(message);
	if ((err = file->Read(message, (size_t)message_size)) < 0)
		return err;

	cur = end = 0;

	while (1) {
		// filter cr-lf (even though BeMail doesn't support it)
		if ((separator == '\r') && (end + 1 < message_size) && (message[end+1] == '\n'))
			end++;
		cur = end + 1;

		for (end = cur;end<message_size;end++) {
			separator = message[end];

			if ((separator == '\r') || (separator == '\n'))
				break;
			
			// cheesy way to identify non-mail files
			if (!isprint(separator) && !isspace(separator))
				return NOT_MAIL;
		}

		if (end >= message_size)
			break;

		message[end] = 0;

		// end of header?
		if (end == cur) {
			if (is_mail) {
				// again, filter cr-lf
				if ((separator == '\r') && (end + 1 < message_size) && (message[end+1] == '\n'))
					end++;
				int32 header_end = end + 1;
				int32 body_len = message_size - header_end;
				file->WriteAttr(B_MAIL_ATTR_HEADER, B_INT32_TYPE, 0, &header_end, sizeof(int32));
				file->WriteAttr(B_MAIL_ATTR_CONTENT, B_INT32_TYPE, 0, &body_len, sizeof(int32));
				file->WriteAttr(B_MAIL_ATTR_STATUS, B_STRING_TYPE, 0, "Read", 5);
	
				// place this here so non-mail files don't get file types munged
				BNodeInfo node(file);
				node.SetType(B_MAIL_TYPE);
			}
			break;
		}

#define HANDLE_FIELD(attr,str) \
		if (!strncasecmp(message+cur, str, sizeof(str) - 1)) { \
			is_mail = true; \
			file->WriteAttr(attr, B_STRING_TYPE, 0, \
				message + cur + sizeof(str) - 1, \
				end - (cur + sizeof(str) - 1) + 1); \
		}
		
		HANDLE_FIELD(B_MAIL_ATTR_TO, FIELD_TO);
		HANDLE_FIELD(B_MAIL_ATTR_REPLY, FIELD_REPLY);
		HANDLE_FIELD(B_MAIL_ATTR_SUBJECT, FIELD_SUBJECT);
		HANDLE_FIELD(B_MAIL_ATTR_PRIORITY, FIELD_PRIORITY);
		HANDLE_FIELD(B_MAIL_ATTR_MIME, FIELD_MIME);

		if (!strncasecmp(message+cur, FIELD_FROM, sizeof(FIELD_FROM) - 1)) {
			is_mail = true;
			file->WriteAttr(B_MAIL_ATTR_FROM, B_STRING_TYPE, 0,
				message + cur + sizeof(FIELD_FROM) - 1,
				end - (cur + sizeof(FIELD_FROM) - 1) + 1);

			int32 start = cur + sizeof(FIELD_FROM) - 1;
			for (int32 index=start;index<end;index++) {
				if (message[index] == '<') {
					if ((index > start) && 
						((message[index-1] == ' ') ||
						 (message[index-1] == '\t')))
						index--;
					message[index] = 0;

					if (message[start] == '"') {
						while (message[--index] != '"')
							;
						if (index > ++start)
							message[index] = 0;
						file->WriteAttr(B_MAIL_ATTR_NAME, B_STRING_TYPE, 0,
							message + start, index - start + 1);
						break;
					} else {
						file->WriteAttr(B_MAIL_ATTR_NAME, B_STRING_TYPE, 0,
							message + start, index - start + 1);
						break;
					}
				}
			}
		}

		if (!strncasecmp(message+cur, FIELD_WHEN, sizeof(FIELD_WHEN) - 1)) {
			is_mail = true;
			time_t when = parsedate(&message[cur + sizeof(FIELD_WHEN) - 1], time(NULL));
			if (when == -1)
				when = 0;
			file->WriteAttr(B_MAIL_ATTR_WHEN, B_TIME_TYPE, 0,
				&when, sizeof(when));
		}
	}

	return (is_mail) ? B_NO_ERROR : NOT_MAIL;
}

status_t handle_file(const char *name, BTextView *view)
{
#define _PRINT_(a) {if (view) view->Insert(a); else printf(a);}
	BFile file;
	status_t err;
	const char *cur, *last = name;
	for (cur=name;*cur;cur++)
		if (*cur == '/')
			last = cur+1;
	_PRINT_("Processing ");
	_PRINT_(last);
	_PRINT_(": ");
	if ((err = file.SetTo(name, B_READ_WRITE)) != B_NO_ERROR) {
		_PRINT_("Error opening file\n");
		return err;
	}
	if ((err = fix_mail_attr(&file)) != B_NO_ERROR) {
		if (err == NOT_MAIL) {
			_PRINT_("Not a mail file\n");
		} else {
			_PRINT_("General file error :(\n");
		}
		return err;
	} else {
		_PRINT_("Done\n");
	}
	return B_NO_ERROR;
}

void process_refs(entry_ref dir_ref, BMessage *msg, void *)
{
	BWindow *window = new BWindow(BRect(100,100,500,300),
		"Fix Mail Attributes", B_DOCUMENT_WINDOW, 0);
	BTextView *view = new BTextView(BRect(0,0,400 - B_V_SCROLL_BAR_WIDTH,200 - B_H_SCROLL_BAR_HEIGHT), "",
		BRect(0,0,200,200), B_FOLLOW_ALL_SIDES, B_WILL_DRAW);
	BScrollView *scroll = new BScrollView("", view, B_FOLLOW_ALL, 0, true, true);
	BFont font = *be_plain_font;
	font.SetFamilyAndStyle("Zurich", "Roman");
	font.SetSize(12);
	view->SetFont(&font);
	view->MakeResizable(true);
	view->MakeEditable(false);
	window->AddChild(scroll);
	
	window->Show();

	entry_ref file_ref;
	int i;
	for (i=0;msg->FindRef("refs", i, &file_ref) == B_NO_ERROR;i++) {
		BPath path;
		BEntry entry(&file_ref);
		entry.GetPath(&path);
		window->Lock();
		handle_file(path.Path(), view);
		window->Unlock();
	}

	if (i == 0) {
		window->Lock();
		view->Insert("No files selected!\n");
		window->Unlock();
	}
}

int main(int argc, char **argv)
{
	if (argc == 1) {
		printf(	"fixmattr: restores email attributes\n"
				"usage: %s [file1 [file2 [...]]]\n", *argv);
		return 1;
	}

	for (int i=1;i<argc;i++)
		handle_file(argv[i], NULL);

	return 0;
}
