// BookwormConverter.cpp
// Copyright 1997 by Brad Ackerman.  All Rights Reserved.
//
// Implements conversion features.

#include <Directory.h>
#include <Entry.h>
#include <File.h>
#include <Looper.h>
#include <Path.h>
#include <StorageDefs.h>
#include <TypeConstants.h>
#include <mlocale.h>
#include "BookwormConverter.h"
#include "BookwormConsts.h"
#include "BookwormWindow.h"

BookwormConverter::BookwormConverter(BookwormWindow *parent)
				: BLooper("Bookworm: Converter thread")
{
	// save pointer to our parent window for passing status messages
	parentWindow = parent;
	Run();
}

void BookwormConverter::MessageReceived(BMessage *event) {
	BMessage	*msg;
	entry_ref	*sourceRef, *destDirRef;
	char		*destName;
	
	switch (event->what) {
		case PROCESS_HTML:
			// got a file! prompt for destination.
			msg = new BMessage(event);
			msg->what = BOOKMARK_DEST_SPECIFIED;
			parentWindow->savePanel->SetMessage(msg);
			parentWindow->savePanel->Show();
			delete msg;
			break;
		case BOOKMARK_DEST_SPECIFIED:
			// we have open and save panel results.
			// do it!
			sourceRef = new entry_ref;
			destDirRef = new entry_ref;
			event->FindRef("ref", sourceRef);
			event->FindRef("directory", destDirRef);
			event->FindString("name", &destName);
			DoHTMLConversion(sourceRef, destDirRef, destName);
			delete sourceRef;
			delete destDirRef;
			break;
		default:
			inherited::MessageReceived(event);
			break;
	}
}

void BookwormConverter::DoHTMLConversion(entry_ref *sourceRef,
							entry_ref *destDirRef, const char *destName) {
	BEntry sourceEntry(sourceRef, true), *tempEntry;
	BPath path;
	BFile sourceFile(&sourceEntry, B_READ_ONLY);
//	FILE *sourceFile;
	BDirectory parentOfDestDir(destDirRef), destDir;
	char *sourceBuf;
	ssize_t amtRead;
	off_t sizeOfSource;
	status_t error;
	char errorText[40];

	// send source path to status console
	parentWindow->PrintStatusMsg("Bookmark file selected: ");
	sourceEntry.GetPath(&path);
	parentWindow->PrintStatusMsg(path.Path());
	parentWindow->PrintStatusMsg("\n");
	
	// create our destination directory
	if (!(error = parentOfDestDir.CreateDirectory(destName, &destDir))) {
		// directory successfully created!
		// print path of destination directory
		tempEntry = new BEntry;
		destDir.GetEntry(tempEntry);
		tempEntry->GetPath(&path);
		parentWindow->PrintStatusMsg("Destination directory created: ");
		parentWindow->PrintStatusMsg(path.Path());
		parentWindow->PrintStatusMsg("\n");
		
		// read source HTML into memory
		sourceFile.GetSize(&sizeOfSource);
		sourceBuf = new char[sizeOfSource+1];
		amtRead = sourceFile.Read(sourceBuf, sizeOfSource);
		sourceBuf[amtRead] = '\0';
		
		// loop through the file, creating directories as we go.
		
		char url[256], descr[256];
		int i = 0, j;
		BFile *urlFile;
		BNode *urlNode;
		BNodeInfo *urlInfo;
		
		while (MatchString(sourceBuf, "<A HREF=", i)) {
			// we've found a URL, copy it.
			j = 0;
			i++;		// get i off the opening quote
			do
				url[j++] = sourceBuf[i++];
			while (url[j-1] != '"');
			url[j-1] = '\0';
				// now get its description.
			LookFor(sourceBuf, '>', i);			// put i at closing &gt;
			j = 0;
			i++;								// put i on start of description
			do
				descr[j++] = sourceBuf[i++];
			while (descr[j-1] != '<');
			descr[j-1] = '\0';
			
			// make a URL file in our folder.
			urlFile = new BFile;
			destDir.CreateFile(descr, urlFile, false);
			
			// give it attributes
			urlFile->WriteAttr("url", B_STRING_TYPE, 0, url, strlen(url));
			urlFile->WriteAttr("title", B_STRING_TYPE, 0, descr, strlen(descr));
			
			// change its type to bookmark
			urlNode = new BNode(&destDir, descr);
			urlInfo = new BNodeInfo(urlNode);
			urlInfo->SetType("application/x-vnd.Be-bookmark");
				
			// delete our objects
			delete urlInfo;
			delete urlNode;
			delete urlFile;
				
			// print status
//			parentWindow->PrintStatusMsg("Bookmarked: ");
//			parentWindow->PrintStatusMsg(url);
//			parentWindow->PrintStatusMsg("\n");
			parentWindow->PrintStatusMsg(".");
		}
		parentWindow->PrintStatusMsg("\nThat's all, folks!");		
	
	} else {
		// failed to create destination directory, print error
		parentWindow->PrintStatusMsg("Error! Couldn't create directory. Code: ");
		sprintf(errorText, "%d\n", error);
		parentWindow->PrintStatusMsg(errorText);
	} 
}

bool LookFor(char *buffer, char item, int &curPos) {
	while (1) {
		if (buffer[curPos] != '\0') {
			if (buffer[curPos] == item) {
				return true;
			}
		} else
			return false;
		curPos++;
	}	
}

bool MatchString(char *buffer, const char *str, int &curPos) {
	int length = strlen(str), i;
	bool done, endOfBuf, aborted;
	
	if (LookFor(buffer, str[0], curPos)) {
		// got the first character.  Now match the rest.
		done = false;
		endOfBuf = false;
		aborted = false;
		for (i = 0; (i < length) && !done; i++) {
			if (buffer[curPos] == '\0') {
				done = true;
				endOfBuf = true;
				aborted = true;
			} else
				if (!(buffer[curPos] == str[i] || buffer[curPos] == toupper(str[i])
						|| buffer[curPos] == tolower(str[i]))) {
					done = true;	// no match, abort.
					aborted = true;
				} else
					curPos++;
		}
		
		if (endOfBuf)
			return false;			// out of buffer
		else if (!aborted)
			return true;			// match found.
		else
			return MatchString(buffer, str, curPos);	// tail recursion!
	} else
		return false;				// out of buffer
} // MatchString