// RingBuffer and sample rate converter Class for AMP
// (C) 1997 Andy Lo A Foe, arloafoe@cs.vu.nl

#include "Converter.h"

#define SPATIE	8192
#define MARKER	256

Converter::Converter(int32 size)
{
	bytes_in_buf = 0;
	buf_size = size;
	in_speed = out_speed = 44100;	// Default of 44.1Khz
	in_stereo = out_stereo = TRUE;	// stereo stream
	waiting_for = FALSE;
	
	write_index = read_index = 0;

	mutex = create_sem(1, "mutex_semaphore");
	full = create_sem(0, "full_semaphore");
	
	if ((buffer_data = new int32[(buf_size/MARKER)+1])==NULL) {
		printf("Error allocating buffer_data\n");
		return;
	}
		
	if ((buffer = new char[buf_size]) == NULL) {
		printf("Error allocating buffer\n");
		return;
	} else {
		//printf("Allocated %d bytes\n", buf_size);
		memset(buffer,0,buf_size);
	}
	
	Done(FALSE);
	
}


Converter::~Converter()
{
	delete_sem(full);
	delete_sem(mutex);
	
	delete buffer;
	delete buffer_data;
}


void Converter::SetInStream(int32 speed, bool stereo)
{
	in_speed = speed;
	in_stereo = stereo;
}


void Converter::SetOutStream(int32 speed, bool stereo)
{
	out_speed = speed;
	out_stereo = stereo;
}


void Converter::Write(char *buf, int32 size, int32 data)
{	

	if (acquire_sem(mutex) != B_NO_ERROR)
		return;
	
	if ((buf_size-bytes_in_buf) <= SPATIE) {
		waiting_for = TRUE;
		release_sem(mutex);
		acquire_sem(full);
		if (acquire_sem(mutex) != B_NO_ERROR)
			return;	
	}
		
	if ((buf_size-write_index) < size) {
		int32 written = buf_size-write_index;
		memcpy(buffer+write_index, buf, written);
		for (int i=0; i < written; i+=MARKER) {
			buffer_data[(write_index+i) / MARKER] = data;
		}
		memcpy(buffer, buf+written, size-written);
		write_index=size-written;
		for (int i = 0; i < write_index; i+=MARKER) {
			buffer_data[i / MARKER] = data;
		}
	} else {
		memcpy(buffer+write_index, buf, size);
		for (int i=0; i < size; i+=MARKER) {
			buffer_data[(write_index+i) / MARKER] = data;
		}	
		write_index+=size;
	}
	
	bytes_in_buf+=size;
	release_sem(mutex);
}


void Converter::Discard()
{
	acquire_sem(mutex);
	if (waiting_for) {
		waiting_for = FALSE;
		//printf("Releasing full in Discard()\n");
		release_sem(full);
	}
	bytes_in_buf = 0;
	read_index = write_index = 0;
	release_sem(mutex);
}
					


bool Converter::Read(char *buf, int32 size)
{
	if (acquire_sem(mutex)!=B_NO_ERROR)
		return FALSE;
		
	if (bytes_in_buf < SPATIE) {
		memset(buf,0,size);
		if (done == TRUE) {
			//printf("Exiting in converter (done = TRUE)\n");
			release_sem(mutex);
			return FALSE;
		}
		release_sem(mutex);
		return  TRUE;		// No blocking
	}

	// Now we are certain there is enough data for the Read to complete
	// so we don't need those extra checks for buffer overrun...
	// Oops, the above statement was too optimistic. It can go wrong,
	// just not in amp :))
		
	float step = (float) in_speed / (float) out_speed;
	
	if (step == 0.0) {
		memset(buf,0,size);
		release_sem(mutex);
		return TRUE;
	}
	
	int32 buf_index = 0;
	int32 tmp_index = 0;
	
	if ((in_stereo && out_stereo) && (in_speed == out_speed)) {
		int32 *out_buf = (int32*)buf;
		int32 *in_buf =  (int32*)buffer;
		
		while (buf_index < size>>2) {					
			if (read_index >= buf_size>>2)	{ // Loop
				read_index = 0;
			}
			out_buf[buf_index] = in_buf[read_index];
			buf_index++;
			read_index++;
		}
		if (read_index == buf_size>>2) {
			read_index = 0;
		}
		bytes_in_buf-=size;
	} else if (in_stereo && out_stereo) {
		int32 *out_buf = (int32*)buf;
		int32 *in_buf =  (int32*)buffer;
		while (buf_index < size>>2) {
			tmp_index = (int32) (step*(float)buf_index);
			
			if ((read_index+tmp_index) >= buf_size>>2) {
				read_index = -tmp_index;
			}
			out_buf[buf_index] = in_buf[read_index + tmp_index];
			buf_index++;
			
		}
		read_index += tmp_index;
		bytes_in_buf -= (tmp_index<<2); 		
	}  else if (!in_stereo && out_stereo) {
		short *out_buf = (short*)buf;
		short *in_buf =  (short*)buffer;
		
		while (buf_index < size>>1) { 
			tmp_index = (int32) (step*(float)(buf_index>>1));
				
			if ((read_index+tmp_index) >= buf_size>>1) {
				read_index = -tmp_index;
			}
			out_buf[buf_index] = in_buf[read_index + tmp_index];
			out_buf[buf_index+1] = out_buf[buf_index];	// Make an extra copy
			buf_index+=2;
		}
		read_index += tmp_index; 
		bytes_in_buf -= (tmp_index<<1);
	} else {
		printf("The BeOS does not support mono output... Sorry\n");
		exit(1);
	}
	if (waiting_for && (buf_size-bytes_in_buf) > SPATIE) {
		waiting_for = FALSE;
		//printf("Releasing full\n");
		release_sem(full);
	}
	release_sem(mutex); 
	return TRUE;		
}


int32 Converter::Data()
{
	return buffer_data[((read_index << ((!in_stereo && out_stereo) ? 1 : 2)) / MARKER)];
}


void Converter::fillData(int32 data)
{
	for (int i=0; i < (buf_size/MARKER)+1; i++) {
		buffer_data[i] = data;
	}
}
