/* AmTrackView.h
 * Copyright (c)1998-2000 by Eric Hackborn.
 * All rights reserved.
 *
 * This class is the REQUIRED superclass to any overviews returned by an
 * AmViewFactory.  (It is currently required solely because of the function
 * SeqVersionView::RemoveTrack() -- this function runs through its collection of
 * views and finds one with the given track.  We should think of a way to remove
 * this dependency)  When using this as a superclass, subclasses
 * get a little bit of convenience behaviour:
 *      1.  They will automatically store an AmTrack and AmTimeConverter,
 * the two basic objects needed for track drawing.  Both of these objects get set
 * through a BMessage of type ARPMSG_POPULATETRACKVIEW.  Any one who doesn't
 * subclass will most likely want to handle this message themselves.  Note that
 * anyone who does subclass and intercepts this message will likely want to pass
 * it along to this class, and let it handle it, as well.
 *      2.  They will know the basics of how to deal with notification about any
 * changes in the events stored in their track.  NOTE:  Currently, in order to
 * hear about these changes, subclasses need to add themselves as a dependent of
 * DEP_EVENT_CHANGE for their track.  Subclasses need to do this because some
 * subclasses might rather add themselves as a dependent of DEP_EVENT_CHANGE for
 * the version their track is in, if they want to hear about every event coming
 * into and out of the track.
 *
 * Subclasses are required to do three things to take advantage of this class:
 *      1.  Register themselves as a dependent of DEP_EVENT_CHANGE with their track
 * or their track's version.
 *      2.  Override IsInterestingEvent() to return true for every type of event that
 * the subclass knows how to (and wants to) draw.
 *      3.  Override RectForEvent() to return a BRect for every type of event that the
 * subclass answered true to in IsInterestingEvent().
 *
 * Subclasses can optionaly override the following functions:
 *      1.  They automatically handle a double-click by sending off an
 * ARPMSG_DOUBLECLICKTRACK message to the window.  The default is to let the window
 * handle double click behaviour, although subclasses can do whatever they like by
 * overriding HandlePrimaryDoubleClick().  The message that is sent can be modified
 * by overriding AsMessage().
 *      2.  They automatically handle a secondary by sending off an
 * ARPMSG_SECONDARYCLICKTRACK message to the window.  The default is to let the window
 * handle double click behaviour, although subclasses can do whatever they like by
 * overriding HandleSecondaryClick().  The message that is sent can be modified
 * by overriding AsMessage().
 *
 * This code is not public domain, nor freely distributable.
 * Please direct any questions or requests to Eric Hackborn,
 * at <hackborn@angryredplanet.com>.
 *
 * ----------------------------------------------------------------------
 *
 * Known Bugs
 * ~~~~~~~~~~
 *
 *	- None!  Ha ha!
 *
 * ----------------------------------------------------------------------
 *
 * History
 * ~~~~~~~
 * 05.17.00		hackborn
 * Updated for new MIDI domain.
 *
 * 11.15.98		hackborn
 * Created this file
 */

#ifndef AMPUBLIC_AMTRACKVIEW_H
#define AMPUBLIC_AMTRACKVIEW_H

#include <be/app/Message.h>
#include <be/interface/View.h>
#include <be/support/String.h>
#include "ArpViewsPublic/ArpViewDefs.h"
#include "ArpViews/ArpBackground.h"
#include "AmPublic/AmSongObserver.h"
#include "AmKernel/AmPhrase.h"
#include "AmPublic/AmTrackViewI.h"

// MESSAGES I WILL SEND OUT
#define ARPMSG_DOUBLECLICKTRACK		'aDCT'
	// This message will have:
	// STR_AMTRACK, a pointer to an AmTrack
	// (optional) SZ_AMTIME, an int32 that is the time that was clicked.
#define ARPMSG_SECONDARYCLICKTRACK	'aSCT'
	// This message will have:
	// STR_AMTRACK, a pointer to an AmTrack
	// (optional) SZ_AMTIME, an int32 that is the time that was clicked.

// A String to identify the name of the factory in the population message.
#define STR_FACTORY_SIGNATURE	"FactorySignature"
// A string used to identify the name of the view in the population message.
#define STR_FACTORY_VIEW_NAME	"FactoryViewName"

// A string to identify my class (when being passed in a BMessage, for example)
#define STR_AMTRACKVIEW			"AmTrackView"

/******************************************************************
 * AM-TRACK-VIEW
 * This is the abstract superclass for any views that look at or
 * modify track data.  Currently, any views modifying data also
 * require that the window they are in subclass from AmEditingStateI,
 * below.  This design will probably change.
 ******************************************************************/
class AmTrackView : public BView,
					public AmSongObserver,
					public AmTrackViewI
{
public:
	AmTrackView(AmSongRef songRef,
				BRect frame,
				const char* name,
				uint32 resizeMask,
				uint32 flags);
	virtual ~AmTrackView();

	/* Subclasses can plug into drawing on the background by
	 * supplying background objects.  This method adds the
	 * background as the tail of any existing backgrounds.
	 */
	void				AddBackground(ArpBackground* background);

	/*---------------------------------------------------------
	 * AM-TRACK-VIEW-I INTERFACE
	 *---------------------------------------------------------*/
	virtual void		SetTrackRef(AmTrackRef trackRef);
	virtual void		SetTimeConverter(const AmTimeConverter* mtc);
	virtual void		SetFactoryKey(const BString& signature, const BString& name);
	virtual AmTrackRef	TrackRef() const;

	virtual void MessageReceived(BMessage* msg);
	// Handle the mouse clicking.
	virtual	void MouseDown(BPoint where);
	/* Default to 0, 0.  Not sure why, that's what the old ArpView
	 * class was doing so I'm maintaining it.
	 */
	virtual	void GetPreferredSize(float *width, float *height);

	/* Subclasses should answer with a BMessage containing any parameters
	 * that they want to be persistant.  Note that subclasses do not own
	 * the BMessage they answer -- it will be deleted by someone else.
	 */
	virtual BMessage* ConfigurationData()	{ return 0; }
	/* Answer the signature of the factory that created me.  The string
	 * returned is owned by me, clients should not modify it.
	 */
	BString& FactorySignature()				{ return mFactorySignature; }
	/* Answer the name that the factory that created me calls me.  The string
	 * returned is owned by me, clients should not modify it.
	 */
	BString& ViewName()						{ return mViewName; }

protected:
	AmTrackRef				mTrackRef;
	const AmTimeConverter*	mMtc;
	ArpBackground*			mHeadBackground;
	
	// Default behaviour is to pass myself AsMessage() on to my Window().
	virtual void HandlePrimaryDoubleClick();
	// Default behaviour is to pass myself AsMessage() on to my Window().
	virtual void HandleSecondaryClick();

	/* Answer myself as a BMessage.  The default implementation is to
	 * answer a BMessage with my track and the time currently being
	 * clicked on.  The what value is undefined.  0 will be answered
	 * if anything goes wrong.  Callers are responsible for deleting
	 * the BMessage.
	 */
	virtual BMessage* AsMessage();
	// I store the name of the factory that created me, along with the
	// name that is sent to that factory to instantiate me.  This should be
	// sent in by the population message.
	BString						mFactorySignature,
								mViewName;

private:
	typedef	BView		inherited;
};

/******************************************************************
 * AM-EDITING-STATE-I
 * This is a convenience class to cover a poor design decision in
 * the AmTrackView:  The window holding that view must also
 * inherit from this state class, to allow the track view a means of
 * requesting information from its window.
 ******************************************************************/
class AmSelectionsI;

class AmEditingStateI
{
public:
	/* Answer a selections object, or 0 if there isn't one.  Never
	 * delete this object -- if you want to clear the selections,
	 * send in 0 to SetSelections().
	 */
	virtual AmSelectionsI*	Selections() const = 0;
	/* Whatever list is sent into this method becomes the property
	 * of this class -- don't delete it, that will be handled
	 * automatically.
	 */
	virtual void SetSelections(AmSelectionsI* selections) = 0;
	/* Answer the current saturation value for the editing state.
	 * There is only on saturation value right now, and it determines
	 * how visible the secondary notes are.
	 */
	virtual float			Saturation() const = 0;

	/* Info views can use this message to communicate with their
	 * corresponding data views.
	 */
	virtual void PostMessageToDataView(BMessage& msg, view_id fromView) = 0;
	/* Data views can use this message to communicate with their
	 * corresponding info views.
	 */
	virtual void PostMessageToInfoView(BMessage& msg, view_id fromView) = 0;
};

#endif 
