/* AmSelectionsI.h
 * Copyright (c)2000 by Angry Red Planet and Eric Hackborn.
 * All rights reserved.
 *
 * This code is not public domain, nor freely distributable.
 * Please direct any questions or requests to Eric Hackborn,
 * at <hackborn@angryredplanet.com>.
 *
 * Known Bugs
 * ~~~~~~~~~~
 *
 *	- None.  Ha, ha!
 *
 * ----------------------------------------------------------------------
 *
 * History
 * ~~~~~~~
 * 2000.05.31		hackborn
 * Created this file.
 */ 
 
#ifndef AMPUBLIC_AMSELECTIONSI_H
#define AMPUBLIC_AMSELECTIONSI_H

#include <be/support/SupportDefs.h>
#include "AmPublic/AmEvents.h"
#include "AmPublic/AmRange.h"
class AmFilterHolderI;
class AmPhrase;
class AmPhraseEvent;
class AmTrack;

/***************************************************************************
 * AM-SELECTIONS-I
 * This object stores a collection of selected events.  Since each event
 * needs to store a container in addition to the actual event, a standard
 * AmPhrase could not be used.  Also, an AmPhrase would be misleading:  Since
 * this collection doesn't hear about changes in events, then the events
 * could have their times changed, but that wouldn't be reflected here.
 ****************************************************************************/
class AmSelectionsI
{
public:
	virtual ~AmSelectionsI()	{ }

	/* Use this to get a new instance of a selections object.
	 */
	static AmSelectionsI* NewSelections();

	/* Answer the time range of the selections.  These values are probably being
	 * computed, so use them sparingly.  The range will not be valid if I have
	 * no events.
	 */
	virtual AmRange		TimeRange() const = 0;
	/* Answer the number of valid events.
	 */
	virtual uint32		CountEvents() const = 0;
	
	/* Answer the event and container at the supplied index.  Return B_OK if
	 * everything went well, otherwise B_ERROR.
	 */
	virtual status_t	EventAt(uint32 index,
								AmEvent** event,
								AmPhraseEvent** container) const = 0;
	/* Answer true if I currently contain the supplied event.
	 */
	virtual bool		IncludesEvent(	const AmEvent* event,
										const AmPhraseEvent* container) const = 0;
	virtual bool		IncludesEvent(	const AmEvent* event,
										const AmPhrase* container) const = 0;
	/* Answer true if I contain any events of the supplied type.
	 */
	virtual bool		IncludesType(AmEvent::EventType type) const = 0;
	/* Add a new event to this collection.
	 */
	virtual status_t	AddEvent(	AmEvent* event,
									AmPhraseEvent* container) = 0;
	/* Add every event in the phrase to this collection.
	 */
	virtual status_t	AddPhrase(AmPhraseEvent* container) = 0;
	
	/* This is used only in special situations -- the list might contain
	 * events that are no longer valid, this method will remove them.  Generally,
	 * only classes that actually store the selections need to do this.
	 */
	virtual void		Scrub() = 0;


	/*---------------------------------------------------------
	 * CONVENIENCE
	 * These methods are not strictly related to the business of
	 * being a selections object, but are based on data the
	 * selections hold and are operations that multiple clients
	 * use.
	 *---------------------------------------------------------*/
	/* Convert myself into a list suitable for playback and answer
	 * the head.  Typically, a client would just pass this list off to
	 * a transport for playback.  If a client does something else,
	 * though, then they're responsible for deleting these events.
	 *
	 * The track can be NULL, but if it is, the filters won't receive
	 * an am_filter_param object, so they can't do any tempo or signature
	 * baed processing.
	 */
	virtual AmEvent*	AsPlaybackList(const AmTrack* track, AmFilterHolderI* filterHolder) const = 0;
};

/***************************************************************************
 * AM-ARRANGE-SELECTIONS-I
 * This object stores a rectangular range area -- a time range and a series
 * of track indexes -- for a single song.
 * I would have liked to reuse the AmSelectionsI object to get this behaviour,
 * but for now the arrange window selection is different enough to justify
 * its own object.
 ****************************************************************************/
class AmArrangeSelectionsI
{
public:
	virtual ~AmArrangeSelectionsI()		{ }

	/* Use this to get a new instance of a selections object.
	 */
	static AmArrangeSelectionsI* New();

	virtual AmRange		TimeRange() const = 0;
	virtual void		SetTimeRange(AmRange range) = 0;

	virtual void		AddTrack(uint32 index) = 0;
	virtual void		RemoveTrack(uint32 index) = 0;
	virtual bool		IncludesTrack(uint32 index) const = 0;
	/* Answer the number of tracks that are included in this selection.
	 */
	virtual uint32		CountTracks() const = 0;
	/* Answer the track at the supplied index.  If the index is
	 * greater than CountTracks(), answer 0.
	 */
	virtual uint32		TrackAt(uint32 index) const = 0;
	/* Answer true if neither a range nor any tracks have been selected.
	 */
	virtual bool		IsEmpty() const = 0;
	/* Answer true if no range has been selected.
	 */
	virtual bool		RangeIsEmpty() const = 0;
	/* Answer true if no tracks have been selected.
	 */
	virtual bool		TracksAreEmpty() const = 0;
	
	virtual AmArrangeSelectionsI* Copy() const = 0;
};

#endif 

