/* AmDefs.h
 * Global definitions for the ARP MIDI classes.
 */

#ifndef AMPUBLIC_AMDEFS_H
#define AMPUBLIC_AMDEFS_H

#include <SupportDefs.h>

// In 4/4 time:
// Measure 1, Beat 1 = 0,
// Measure 1, Beat 2 = 6720,
// Measure 2, Beat 1 = 26880, etc
// Why this particular value?  Well,
// 6720 == 2*2*2*2*2*2*3*5*7
#define PPQN	(6720)

// Type for an event time.  Using the above factor, this gives us about
// 560,000 4-beat measures of music.
typedef int64 AmTime;
// A string to identify my time (when being passed in a BMessage, for example)
#define SZ_AMTIME			"AmTime"

// Covers for using AmTime values in messages
status_t add_time(BMessage& msg, const char* name, AmTime timeVal);
status_t find_time(const BMessage& msg, const char* name, int32 i, AmTime* timeVal);
status_t find_time(const BMessage& msg, const char* name, AmTime* timeVal);

// Covers for using AmTime values in messages, which are independent of
// the current PPQN.
status_t add_rel_time(BMessage& msg, const char* name, AmTime timeVal);
status_t find_rel_time(const BMessage& msg, const char* name, int32 i, AmTime* timeVal);
status_t find_rel_time(const BMessage& msg, const char* name, AmTime* timeVal);

/* These are the available types of views.
 *		ARRANGE_VIEW is the view used in the Arrange window.
 *		PRI_VIEW is the view used in the top half of the
 *				track edit window.
 *		SEC_VIEW is the view used in the bottom half of the
 *				track edit window.
 */
enum TrackViewType {
	ARRANGE_VIEW,
	PRI_VIEW,
	SEC_VIEW,
	
	_NUM_VIEW
};

	/*---------------------------------------------------------
	 * CHANGE NOTIFICATION
	 * Change notification happens from AmTrack and AmSong
	 * implementation classes.  The available messages an
	 * observer can subscribe to are described in those classes.
	 * There are also a few details in common between the two
	 * that are described here.
	 *---------------------------------------------------------*/

/* Whenever a song or track sends out a prechange or postchange notice,
 * the client that caused the change notice has the option of supplying
 * itself as the sender argument in the message.
 */
#define SZ_SENDER		"sender"
/* When clients are notified about changes to events in songs and tracks
 * via the PRECHANGE_* and POSTCHANGE_* observer messages, the message
 * will also contain the additional field
 *		"status"	An int32 that describes the state of the event.
 */
#define SZ_STATUS		"status"

enum AmStatus {
	AM_NO_STATUS			= 0,	// This should never actually occur, it's more a safety measure
	AM_CHANGED,						// A catch-all
	AM_ADDED,						// The event has been added to the domain object
	AM_REMOVED						// The event has been removed from the domain object
};

/* Various objects in the system can be referenced by their ID's.  These
 * defines were previously in the headers for the relevant classes, but
 * that just caused too many nuisances with header files being included
 * solely for the id typedef.
 */
typedef void*			event_id;
typedef void*			filter_id;
typedef void*			song_id;
typedef void*			track_id;

/* This is the signature for the default view factory.  This factory can
 * always be accessed by calling AmGlobals().FactoryNamed( BString(DEFAULT_FACTORY_SIGNATURE) );
 */
#define DEFAULT_FACTORY_SIGNATURE	"arp:Default"

	/*---------------------------------------------------------
	 * ERRORS
	 *---------------------------------------------------------*/

#define AM_ERROR_BASE				B_ERRORS_END + 0x1000

enum {
	ARPERR_PARTIAL_OPERATION = AM_ERROR_BASE
};

	/*---------------------------------------------------------
	 * IMAGE CONSTANTS.  Use the ImageManager() to access the
	 * actual images.
	 *---------------------------------------------------------*/

#define SLICE_INFO_TOP_BG			"Slice Info Top BG"
#define SLICE_PROPERTY_MENU_NORMAL_IMAGE_STR "Slice Property Menu Normal"
#define SLICE_PROPERTY_MENU_PRESSED_IMAGE_STR "Slice Property Menu Pressed"
#define CHANNEL_01_IMAGE_STR		"Channel 01"
#define CHANNEL_02_IMAGE_STR		"Channel 02"
#define CHANNEL_03_IMAGE_STR		"Channel 03"
#define CHANNEL_04_IMAGE_STR		"Channel 04"
#define CHANNEL_05_IMAGE_STR		"Channel 05"
#define CHANNEL_06_IMAGE_STR		"Channel 06"
#define CHANNEL_07_IMAGE_STR		"Channel 07"
#define CHANNEL_08_IMAGE_STR		"Channel 08"
#define CHANNEL_09_IMAGE_STR		"Channel 09"
#define CHANNEL_10_IMAGE_STR		"Channel 10"
#define CHANNEL_11_IMAGE_STR		"Channel 11"
#define CHANNEL_12_IMAGE_STR		"Channel 12"
#define CHANNEL_13_IMAGE_STR		"Channel 13"
#define CHANNEL_14_IMAGE_STR		"Channel 14"
#define CHANNEL_15_IMAGE_STR		"Channel 15"
#define CHANNEL_16_IMAGE_STR		"Channel 16"
#define FILTER_BACKGROUND			"Filter Background"
#define FILTER_ADDON_MISSING		"Filter AddOn Missing"
#define MIDI_PORT_FILTER_IMAGE_STR	"Midi Port Filter"
#define BE_MIDI_SYNTH_FILTER_IMAGE_STR "Be MIDI Synth Filter"
#define PENCIL_TOOL_NORMAL			"Pencil Tool Normal"
#define PENCIL_TOOL_OVER			"Pencil Tool Over"
#define PENCIL_TOOL_PRESSED			"Pencil Tool Pressed"
#define SELECT_TOOL_NORMAL			"Select Tool Normal"
#define SELECT_TOOL_OVER			"Select Tool Over"
#define SELECT_TOOL_PRESSED			"Select Tool Pressed"
#define WAND_TOOL_NORMAL			"Wand Tool Normal"
#define WAND_TOOL_OVER				"Wand Tool Over"
#define WAND_TOOL_PRESSED			"Wand Tool Pressed"
#define ERASER_TOOL_NORMAL			"Eraser Tool Normal"
#define ERASER_TOOL_OVER			"Eraser Tool Over"
#define ERASER_TOOL_PRESSED			"Eraser Tool Pressed"

/*----------------- The standard view factory images ----------*/
#define	TREBLE_CLEF_IMAGE_STR		"treble_clef"
#define	BASS_CLEF_IMAGE_STR			"bass_clef"


	/*---------------------------------------------------------
	 * COLOUR CONSTANTS.  Use Prefs().Color() to access the
	 * actual colours.
	 *---------------------------------------------------------*/

enum {
	AM_CONTROL_BG_C				= 30,
	AM_LCD_C					= 31,

	AM_MEASURE_TOP_BG_C			= 32,
	AM_MEASURE_BOTTOM_BG_C		= 33,
	AM_MEASURE_LEFT_BG_C		= 34,
	AM_MEASURE_RIGHT_BG_C		= 35,
	AM_MEASURE_FG_C				= 36,
	AM_MEASURE_BEAT_C			= 37,
	AM_MEASURE_HIGHLIGHT_C		= 38,

	AM_DATA_BACKDROP_C			= 39,
	AM_DATA_BG_C				= 40,
	AM_DATA_FG_C				= 41,
	AM_HIGH_PRIMARY_EVENT_C		= 42,
	AM_LOW_PRIMARY_EVENT_C		= 43,
	AM_HIGH_SECONDARY_EVENT_C	= 44,
	AM_LOW_SECONDARY_EVENT_C	= 45,
	AM_HIGH_SELECTED_EVENT_C	= 46,
	AM_LOW_SELECTED_EVENT_C		= 47,
	
	AM_ARRANGE_BG_C				= 48,
	AM_ARRANGE_FG_C				= 49,
	
	AM_PIPELINE_C				= 50,
	
	DUPLICATE_FILTER_1_C		= 51,
	DUPLICATE_FILTER_2_C		= 52,
	DUPLICATE_FILTER_3_C		= 53,
	DUPLICATE_FILTER_4_C		= 54,
	DUPLICATE_FILTER_5_C		= 55,
	DUPLICATE_FILTER_6_C		= 56,
	DUPLICATE_FILTER_7_C		= 57,
	DUPLICATE_FILTER_8_C		= 58,
	DUPLICATE_FILTER_9_C		= 59
};

#endif
