//============================================================================
// AddOnListItem.cpp
// © Michael Pieper 21. Mar. 1999
//============================================================================
//
//	This class manages the add-ons ListItem. 
//  It can draw the correct icon and the AddOn-Filename
//  It also displays, if the item is a server or a client
//
//============================================================================

#include "AddOnListItem.h"
#include "AddOnConfigWindow.h"
#include "messages.h"
#include <InterfaceKit.h>		// We want to do some drawing :-)
#include <StorageKit.h>	

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

//============================================================================
// Constructor for the ListItem.
//

AddOnListItem::AddOnListItem( BEntry *entry )
		: BListItem( ) 

{
	char		fn[B_FILE_NAME_LENGTH];
	app_info	apinfo;
	
	BNode		node;
	BNodeInfo	ninf;
	BRect		r(0,0,15,15);
	BEntry		ent;
	entry_ref	ref;
	
	img_id = B_ERROR;				// Setup the img_id for the information that it isn't loaded.
	thr_id = B_ERROR;				// Setup the thr_id for the information, that the AddOn isn't running
	
	if (entry == NULL) {			// Thats the setupmessage. We have to setup all builtin functions
		filename = strdup("NCP");	// The Builtin is called NCP
		be_app->GetAppInfo(&apinfo);// we are interested in our Startingdirectory
		ent.SetTo(&apinfo.ref, true);	// but we can get only the entry_ref of PalmBeach
	} else {
		entry->GetName(fn);
		filename = strdup( fn );	// Lets copy the filename
		entry->GetRef(&ref);		// find the entry_ref to follow the link
		ent.SetTo(&ref, true);		// we must follow the link to get the correct ICON!
	}
	node.SetTo(&ent);				// Lets set the Node.
	ninf.SetTo(&node);				// The Nodeinfo knows the Icon!

	smallicon = new BBitmap( r, B_CMAP8 );
	ninf.GetTrackerIcon(smallicon, B_MINI_ICON);
	
}

//============================================================================
// Destructor for the ListItem.
//

AddOnListItem::~AddOnListItem( void ) {

	delete smallicon;
	free(filename);
}

//============================================================================
// We have to draw our String and the Icon!
//

void AddOnListItem::DrawItem(BView *owner, BRect frame, bool complete) { 

	font_height	fh;
	float 		f;
	rgb_color 	kMedGrey =		ui_color(B_MENU_BACKGROUND_COLOR);
	rgb_color 	kBlackColor =	tint_color(kMedGrey, B_DARKEN_MAX_TINT);
	rgb_color 	kHighlight =	tint_color(kMedGrey, B_HIGHLIGHT_BACKGROUND_TINT);
	
         
	if (IsSelected()) {								// If the Item is selected,
		owner->SetHighColor( kHighlight );			// set the color's
		owner->SetLowColor( kHighlight );
		owner->FillRect( frame ); 					// and fill the rectangular
	} else {
		owner->SetLowColor( owner->ViewColor() );	// otherwise set the lowcolor correct
	}
	
	if (complete) {
		owner->SetHighColor( owner->ViewColor() ); 
		owner->FillRect(frame); 
	}

	owner->MovePenTo(frame.left+2, frame.top);		// Now we want to draw the Bitmap
	owner->SetDrawingMode( B_OP_OVER );				// use the Backgroundcolor for Transparecy
	owner->DrawBitmap(smallicon);					// and draw it

	owner->GetFontHeight(&fh);
	
	f = (Height() - (fh.descent+fh.ascent))/2;
	if (f < fh.descent) f = fh.descent;
	owner->MovePenTo(frame.left+22, frame.bottom-f);			// We have to write the Text
	owner->SetHighColor( IsEnabled() ? kBlackColor : kMedGrey );// Color depending on Enable/Disable
	owner->DrawString(filename);								// And printing

}

//============================================================================
// If the Font is lower then 16 Pixels, then we have to set it to 16
//

void AddOnListItem::Update(BView *owner, const BFont *font) {
	
	BListItem::Update(owner, font);			// Call the normal Update Routine
	if (Height() < 16) SetHeight( 16.0 );	// Is it lower then 16, then set to 16
}

//============================================================================
// Returns the image_id of an addon. If the AddOn isn't in the memory, then it
// will be loaded. The image_id is returned after that.
// The Function Initialise() of the addon is called to initialise some things.
//

image_id AddOnListItem::GetImageId( void ) {
	
	int32		(*Run)( void );			// Pointer to a function which Initialises the AddOn
	status_t	result;

	if (img_id == B_ERROR) {				// The AddOn is loaded, so we can return the img_id
		img_id = load_add_on( filename );	// Otherwise lets load the AddOn and return the img_id
		if (img_id >= B_OK) {
			result = get_image_symbol(img_id, "Initialise", B_SYMBOL_TYPE_TEXT, (void **)&Run);	// Get the RunServer()-Adress
			if (result == B_OK) {
				Run();
			}
		}
	}
	return( img_id );
}

//============================================================================
// Returns the thread_id of an addon. If the AddOn isn't in the memory, then it
// will be loaded. The process is started as a own thread. 
// whicone should be "RunServer" or "RunClient"
//

thread_id AddOnListItem::RunThread( char *whichone ) {
	
	int32		(*Run)(void *);			// Pointer to a function which starts the server
	status_t	result;
	
	if (thr_id >= B_NO_ERROR) return( thr_id );	// the Thread is always running
	
	if ( (img_id = GetImageId()) >= B_NO_ERROR ) {	// Load the Image and Initialise it. If all is correct continue.
		result = get_image_symbol(img_id, whichone, B_SYMBOL_TYPE_TEXT, (void **)&Run);	// Get the RunServer()-Adress
		if (result == B_OK) {
			thr_id = spawn_thread(Run, filename, B_NORMAL_PRIORITY, filename);		// Lets start as a own thread
   			if ((thr_id >= B_NO_ERROR) & (resume_thread(thr_id) >= B_NO_ERROR)) {	// When all is OK, then
				return ( thr_id );													// return the thread ID!
			}
		}
	}
	return( B_ERROR );
}

//============================================================================
// Stops the thread if he is running We are sending him a message PSION_STOP
// and wait 10 Seconds. 

bool AddOnListItem::StopThread( void ) {
	
	if (thr_id < B_NO_ERROR) return ( true );	// The thread isn't running
	send_data(thr_id, PSION_DISC, NULL, 0);		// lets simple send a Msg to the thread!
	thr_id = B_ERROR;							// we have to check, if the thread died..... something has to change here!
		
	return( true );
}

//============================================================================
// Starts the ConfigWindow

void AddOnListItem::ShowConfigWindow( BRect frame ) {
	
	char		titel[55];
	
	if (img_id == B_ERROR) img_id = load_add_on( filename );	// lets load the addon, when its not loaded!
	if (img_id != B_ERROR) {				// All done correct?
	
		sprintf(titel, "%.40s Settings", filename);
		AddOnConfigWindow *win = new AddOnConfigWindow( frame, titel, img_id );
		if (win->HasConfigView()) win->Show();
		else delete win;
	}
}

//============================================================================
// This function returns the Filename
//

char *AddOnListItem::Text( void ) {
	
	return( filename );	
}
