#include <InterfaceKit.h>
#include <StorageKit.h>
#include <AppKit.h>
#include <stdio.h>
#include <string.h>

enum { YES, NO, NONE};
extern const char *copyright= "MIMEset copyright 1998 and written by David Mitchell.  Contact: mitchell@141.com";

class DSampleDropView : public BBox
{
	public:
		DSampleDropView (BRect bounds);
		void MessageReceived(BMessage *msg);
		void MouseMoved(BPoint point, uint32 transit, const BMessage *msg);
		void Draw (BRect update);
	private:
		bool CanAccept (const BMessage *msg);
		rgb_color bc;
		char state;
};

DSampleDropView::DSampleDropView (BRect bounds)
	:BBox (bounds, "SampleMIMEDropZone", B_FOLLOW_TOP | B_FOLLOW_LEFT_RIGHT, B_WILL_DRAW)
{
	SetLabel("Sample File");
	bc = ViewColor();
	state = NONE;
}

void DSampleDropView::MessageReceived (BMessage *m)
{
	if (m->WasDropped()) {
		state = NONE;
		Invalidate();
	};
	
	if (CanAccept(m)) {
		BMessage sm('wdmA');
		entry_ref r;
		if (m->FindRef("refs", &r) == B_OK) {
			BEntry e(&r, true);
			BNode n(&e);
			if (n.InitCheck() == B_NO_ERROR) {
				BNodeInfo ni(&n);
				if (ni.InitCheck() == B_NO_ERROR) {
					char t[B_MIME_TYPE_LENGTH];
					if (ni.GetType(t) == B_NO_ERROR) {
						sm.AddString("type", t);
						BWindow *w = Window();
						w->PostMessage(&sm, w);
						return;
					};
				};
			};
		};
	};
	BView::MessageReceived (m);
}

void DSampleDropView::MouseMoved(BPoint , uint32 t, const BMessage *m)
{
	if (t == B_ENTERED_VIEW && m) {
		if (CanAccept(m)) {
			state = YES;
			Invalidate();
		}
		else {
			state = NO;
			Invalidate();
		}
	} else if (t == B_EXITED_VIEW) {
		state = NONE;
		Invalidate();
	};
}

bool DSampleDropView::CanAccept (const BMessage *m)
{
	if (!m->HasRef("refs", 0))
		return false;  	//No refs at all
	if (m->HasRef("refs", 1))
		return false;	//Too many refs
	
	entry_ref r;
	if (m->FindRef("refs", &r) != B_OK)
		return false;
	BEntry e(&r, true);
	if (e.InitCheck() != B_NO_ERROR)
		return false;
		
	struct stat s;
	if (e.GetStat(&s) != B_NO_ERROR)
		return false;
		
	if (!S_ISREG(s.st_mode))
		return false;

	return true;
}

void DSampleDropView::Draw(BRect r)
{
	BBox::Draw(r);
//	PushState();
	BRect s(Bounds());
	s.InsetBy(15,15);
	SetPenSize(5);
	switch (state) {
		case YES:
			SetHighColor(0,255,0);
			StrokeEllipse(s);
			break;
		case NO:
			SetHighColor(255,0,0);
			StrokeLine(s.LeftTop(),s.RightBottom());
			StrokeLine(s.RightTop(), s.LeftBottom());
			break;
		case NONE:
			SetHighColor(bc);
			break;
	};

//	PopState();
}


class DTargetDropView : public BBox
{
	public:
		DTargetDropView (BRect bounds);
		void MessageReceived(BMessage *msg);
		void MouseMoved(BPoint point, uint32 transit, const BMessage *msg);
		void SetType (const char *type);
		void Draw (BRect update);
	private:
		bool CanAccept (const BMessage *msg);
		char MIMEtype[B_MIME_TYPE_LENGTH];
		void ProcessRef (entry_ref r, bool verified = false);
		void ProcessDirectory (BEntry e, bool verified = false);
		void ProcessFile (BEntry e);
		bool CheckIt (BEntry e);
		rgb_color bc;
		char state;
};

DTargetDropView::DTargetDropView (BRect bounds)
	:BBox (bounds, "TargetMIMEDropZone", B_FOLLOW_BOTTOM | B_FOLLOW_LEFT_RIGHT, B_WILL_DRAW)
{
	MIMEtype[0] = 0;
	SetLabel("Target Files");
	bc = ViewColor();
	state = NONE;
}

void DTargetDropView::MessageReceived (BMessage *m)
{
	if (m->WasDropped()) {
		state = NONE;
		Invalidate();
		if (CanAccept(m)) {
			entry_ref r;
			for (int i = 0; (m->FindRef("refs", i, &r) == B_OK); i++) {
				ProcessRef(r);
			};
		};

	};

	BView::MessageReceived (m);
}

void DTargetDropView::ProcessRef(entry_ref r, bool verified)
{
	BEntry e(&r, true);
	if (e.InitCheck() == B_NO_ERROR) {
		struct stat st;
		if (e.GetStat(&st) == B_NO_ERROR) {
			if (S_ISREG(st.st_mode)) 
				ProcessFile(e);
      		else if (S_ISDIR(st.st_mode)) 
      			ProcessDirectory(e, verified);
      	};
	};
}

void DTargetDropView::ProcessFile(BEntry e)
{
	BNode n(&e);
	if (n.InitCheck() == B_NO_ERROR) {
		BNodeInfo ni(&n);
		if (ni.InitCheck() == B_NO_ERROR) {
			ni.SetType(MIMEtype);
		};
	};
}

void DTargetDropView::ProcessDirectory(BEntry e, bool verified)
{
	return;
	if (!verified) {
		if (!CheckIt(e))
			return;
		verified = true;
	};
		
		
	BDirectory d(&e);
	if (d.InitCheck() == B_NO_ERROR) {
		entry_ref r;
		d.Rewind();
		while (d.GetNextRef(&r) == B_OK)
			ProcessRef(r, verified);
	};
}

bool DTargetDropView::CheckIt(BEntry e)
{
	const directory_which dirs[] = {
		B_APPS_DIRECTORY,
		B_PREFERENCES_DIRECTORY,
		B_BEOS_DIRECTORY,
		B_BEOS_SYSTEM_DIRECTORY,
		B_BEOS_ADDONS_DIRECTORY,
		B_BEOS_LIB_DIRECTORY,
		B_BEOS_SERVERS_DIRECTORY,
		B_BEOS_APPS_DIRECTORY,
		B_BEOS_BIN_DIRECTORY,
		B_BEOS_ETC_DIRECTORY,
		B_BEOS_DOCUMENTATION_DIRECTORY,
		B_BEOS_PREFERENCES_DIRECTORY,
		B_COMMON_SYSTEM_DIRECTORY,
		B_COMMON_ADDONS_DIRECTORY,
		B_COMMON_BOOT_DIRECTORY,
		B_COMMON_FONTS_DIRECTORY,
		B_COMMON_LIB_DIRECTORY,
		B_COMMON_SERVERS_DIRECTORY,
		B_COMMON_BIN_DIRECTORY,
		B_COMMON_ETC_DIRECTORY,
		B_COMMON_DOCUMENTATION_DIRECTORY,
		B_COMMON_SETTINGS_DIRECTORY,
		B_COMMON_DEVELOP_DIRECTORY,
		B_COMMON_LOG_DIRECTORY,
		B_COMMON_SPOOL_DIRECTORY,
		B_COMMON_TEMP_DIRECTORY,
		B_COMMON_VAR_DIRECTORY,
		};
		
	BPath p1, p2;
	BDirectory d, d2;
	BAlert *a;
	BEntry te;
	char buff[256 + 2*B_PATH_NAME_LENGTH];
	e.GetPath(&p2);
	
	
	for (int i = 0; i < 27; i++) {
		if (find_directory(dirs[i], &p1) == B_OK) {
			if (d.SetTo(p1.Path()) == B_OK) {
				te.SetTo(p1.Path());
				if (d.Contains(&e)) {
					sprintf(buff, "The directory you selected \"%s\" is part of the system directory \"%s.\"  Changing the MIME types may damage your system.  Are you sure you want to perform this operation?", p2.Path(), p1.Path());
					a = new BAlert ("Directory Problem", buff, "Cancel" , "OK", (char *)NULL, B_WIDTH_FROM_LABEL, B_WARNING_ALERT);
					int j = a->Go();
					if (j != 1)
						return false;
					else
						return true;
				};
				if (te == e) {
					sprintf(buff, "The directory you selected \"%s\" is a system directory.  Changing the MIME types may damage your system.  Are you sure you want to perform this operation?", p2.Path());
					a = new BAlert ("Directory Problem", buff, "Cancel" , "OK", (char *)NULL, B_WIDTH_FROM_LABEL, B_WARNING_ALERT);
					int j = a->Go();
					if (j != 1)
						return false;
					else
						return true;
				};
				
				if (d.SetTo(p2.Path()) == B_OK) {
					if (d.Contains(&te)) {
						sprintf(buff, "The directory you selected \"%s\" contains the system directory \"%s,\" and possibly other system directories.  Changing the MIME types may damage your system.  Are you sure you want to perform this operation?", p2.Path(), p1.Path());
						a = new BAlert ("Directory Problem", buff, "Cancel" , "OK", (char *)NULL, B_WIDTH_FROM_LABEL, B_WARNING_ALERT);
						int j = a->Go();
						if (j != 1)
							return false;
						else
							return true;
					};
				};
					
			};
		};
	};
	
	
	return true;
}		
	
	
void DTargetDropView::MouseMoved(BPoint , uint32 t, const BMessage *m)
{
	if (t == B_ENTERED_VIEW && m) {
		if (CanAccept(m)) {
			state = YES;
			Invalidate();
		}
		else {
			state = NO;
			Invalidate();
		}
	} else if (t == B_EXITED_VIEW) {
		state = NONE;
		Invalidate();
	};
}


bool DTargetDropView::CanAccept (const BMessage *m)
{
	if (!m->HasRef("refs", 0))
		return false;  	//No refs at all
	if (!MIMEtype[0])
		return false;	//No prototype yet
	
	entry_ref r;
	if (m->FindRef("refs", &r) != B_OK)
		return false;
	BEntry e(&r, true);
	if (e.InitCheck() != B_NO_ERROR)
		return false;
		
	struct stat s;
	if (e.GetStat(&s) != B_NO_ERROR)
		return false;
		
	if (!S_ISREG(s.st_mode))
		return false;


	return true;
}

void DTargetDropView::SetType (const char *t)
{
	strncpy (MIMEtype, t, B_MIME_TYPE_LENGTH);
}


void DTargetDropView::Draw(BRect r)
{
	BBox::Draw(r);
	PushState();
	BRect s(Bounds());
	s.InsetBy(15,15);
	SetPenSize(5);
	switch (state) {
		case YES:
			SetHighColor(0,255,0);
			StrokeEllipse(s);
			break;
		case NO:
			SetHighColor(255,0,0);
			StrokeLine(s.LeftTop(),s.RightBottom());
			StrokeLine(s.RightTop(), s.LeftBottom());
			break;
		case NONE:
			SetHighColor(bc);
			break;
	};

	PopState();
}

class DTypeSetterWin : public BWindow
{
	public:
		DTypeSetterWin ();
		void MessageReceived(BMessage *msg);
		bool QuitRequested();
	private:
		char type[B_MIME_TYPE_LENGTH];
		BStringView *sv, *sv2;
		DSampleDropView *dsdv;
		DTargetDropView *dtdv;
};

DTypeSetterWin::DTypeSetterWin()
	: BWindow (BRect(50,50,275,205), "MIMEset", B_TITLED_WINDOW ,B_NOT_ZOOMABLE|B_NOT_RESIZABLE)
{
	BBox * bv = new BBox(Bounds(), "Background", B_FOLLOW_ALL, B_WILL_DRAW, B_PLAIN_BORDER);
	AddChild(bv);
	dsdv = new DSampleDropView(BRect(5,35,100,130));
	bv->AddChild(dsdv);
	sv = new BStringView(BRect(5,15,220,30),"MIME type view", "Uninitialized", B_FOLLOW_LEFT_RIGHT, B_WILL_DRAW);
	sv->SetAlignment (B_ALIGN_CENTER);
	bv->AddChild(sv);
	dtdv = new DTargetDropView(BRect(125,35,220,130));
	bv->AddChild(dtdv);
	sv2 = new BStringView(BRect(5,130,220,145),"Copyright notice", "Copyright © 1998 David Mitchell",B_FOLLOW_LEFT_RIGHT, B_WILL_DRAW);
	bv->AddChild(sv2);
}

void DTypeSetterWin::MessageReceived (BMessage *m)
{
	switch (m->what) {
		case 'wdmA':
			char *t;
			if (m->FindString("type", &t) == B_NO_ERROR) {
				strncpy (type, t, B_MIME_TYPE_LENGTH);
				sv->SetText(type);
				dtdv->SetType(type);
			};
			break;
		default:
			BWindow::MessageReceived(m);
	};
}


bool
DTypeSetterWin::QuitRequested()
{
 	be_app->PostMessage(B_QUIT_REQUESTED);
	return true;
}
	


int main()
{
	BApplication App("application/x-vnd-WDM-MIMEset");
	
	DTypeSetterWin *w = new DTypeSetterWin();
	

	
	w->Show();
	App.Run();
}