// This is a sample plugin snippet for Eddie
//
// (c) 1998 P.Cisler
//
// Any portion of this code may be used freely to develop Eddie
// plugins 
//
// The plugin illustrates how to implement something like
// electric-C mode. It's not very usefull on itself, the purpose is for
// developers to get the idea and write a proper one based on this
// sample
//
// The plugin sample shows you how to:
//
// 		intercept key down events
//		scan text
//		insert text and use specific undo name
//		adding plugin settings that get stored in Eddies' settigs file
//		adding a preference panel to Eddie's Application Settings
//
// This plugin has no button and does not publish any primitives for
// keyboard shortcuts. Look at the other plugin examples for illustrations
// on how to setup plugins with buttons and shortcuts.

#include <CheckBox.h>
#include <View.h>

#include <stdlib.h>
#include <string.h>

#include "Plugin.h"

class Settings {
	// this class manages the state of the settings
	// also, it handles adding a preference panel into the
	// Application Settings dialog
public:
	Settings(PluginInterface *);

	bool HandleSettingsCommand(PluginInterface *, const char ***);
	void Save(PluginInterface *);
	void AddPreferencePanel(PluginInterface *);

	void Apply();
	void Revert();
	void Defaults();

	bool CanApply() const;
	bool CanRevert() const;
	bool CanDefault() const;

	bool On() const
		{ return on; }

	static Settings *settings;
private:
	// after all this effor the only setting we have here is the
	// ability to turn this plugin on and off, but you get the point
	// hopefully
	bool on;
	bool revertValue;
	BCheckBox *onCheckBox;
};

class SamplePrefsPanel : public PrefsPanel {
	// this class is an interface between Eddie and the plugins
	// settings panel; we will do it as a simple proxy for
	// the actual Settings class, that will do the work
public:
	SamplePrefsPanel(BView *panelView)
		:	PrefsPanel(panelView)
		{}

	virtual void Revert()
		{ settings->Revert(); }

	virtual void Defaults()
		{ settings->Defaults(); }

	virtual void Apply()
		{ settings->Apply(); }

	virtual bool CanApply() const
		{ return settings->CanApply(); }

	virtual bool CanRevert() const
		{ return settings->CanRevert(); }

	virtual bool CanDefault() const
		{ return settings->CanDefault(); }


	void SetTarget(Settings *target)
		{ settings = target; }
	
private:
	Settings *settings;

};

Settings *Settings::settings = 0;

PluginInterface::PluginResult 
PluginMain(PluginInterface::PluginCallSelector selector, PluginInterface *pluginInterface)
{
	PluginInterface::PluginResult result = PluginInterface::kIgnored;

	switch (selector) {
		case PluginInterface::kGetPluginVersion:
			// this is the first selector call a plugin will get
			// Just include the following two lines to let Eddie know the
			// version of plugin API you support and the version you require
			// from Eddie
			//
			// this is the only selector call your plugin is required to support
			//
			pluginInterface->ReturnSupportedPluginVersion();
			pluginInterface->ReturnRequiredPluginVersion();
			result = PluginInterface::kAccepted;
			break;


		case PluginInterface::kInit:
			// a button may allocate any permanent data structures it will
			// need during it's lifetime and initialize it's state

			// this plugin sports settings with it's own preference panel
			// initialize the Settings object, registering it with Eddie
			Settings::settings = new Settings(pluginInterface);

			// ask for any non-default selector calls you may need
			// these include:
			// kButtonDraw		- if you want to draw the button yourself instead of just
			//					  passing an icon
			// kPulse			- if you want to get called periodically
			// kMessage			- if you want to get a peek at every message sent to the
			//					  text view
			// kMouseDown		- if you want to get called every time the mouse is
			//					  pressed over a text view
			// kKeyDown			- if you want to get called every time a key is hit in a
			//					  text view
			// kDocumentChanged	- if you want to get called every time the document changes
			//
			// kKeyDown selector hast to be explicitly requested by a plugin
			pluginInterface->RequireSelectors(PluginInterface::kRequestKeyDownMask);
			result = PluginInterface::kAccepted;
			break;

		case PluginInterface::kArgvPassed:
			// settings reading time; Eddie found a setting item that we
			// registered to handle; we are now being called to process it
			Settings::settings->HandleSettingsCommand(pluginInterface,
				pluginInterface->CurrentArgv());
			result = PluginInterface::kAccepted;
			break;

		case PluginInterface::kGlobalSettingsSaved:
			// settings are being saved; do our part
			Settings::settings->Save(pluginInterface);
			result = PluginInterface::kAccepted;
			break;

		case PluginInterface::kGlobalPrefsShowing:
			// the Application Settings window is being built; add our own
			// page
			Settings::settings->AddPreferencePanel(pluginInterface);
			result = PluginInterface::kAccepted;
			break;

		case PluginInterface::kKeyDown: {
				// handle the KeyDown event
				
				if (!Settings::settings->On())
					// we are not turned on, bail
					break;

				if (pluginInterface->NumKeyBytes() != 1)
					// only interrested in single byte characters
					break;
				char key = *pluginInterface->KeyBytes();
				if (key != '{')
					// in this example only interrested in open curly
					break;

				int32 selStart, selEnd;
				pluginInterface->GetSelection(&selStart, &selEnd);
				if (selStart != selEnd)
					// only do our stuff if selection is empty
					break;

				// count tabs at the beginning of the line we are currently on				
				int32 lineStart = pluginInterface->StartOfLine(selStart);
				int32 count;
				for (count = 0; ; count++)
					if (pluginInterface->CharAt(lineStart++) != '\t')
						break;
						
				const char *stringToInsert = "{\n\t\t\t\t\t\t\t\t\t\t\t\t\t\t\t\t\t"
					"\t\t\t\t\t\t\t\t\t\t\t\t\t\t\t\t\t\t\t\t\t\t\t\t\t\t";

				if (count > strlen(stringToInsert) - 2)
					// the indent level is way to deep, just bail
					break;
				
				// account for inserted curly, newline and extra tab
				count += 3;
				
				// insert the resulting indenting string; the undo for this
				// insert will read Undo Brace Auto Indent
				pluginInterface->Insert(stringToInsert, count, "Brace Auto Indent");

				// we have completely taken care of the keypress, let Eddie know to
				// further disregart the KeyDown event by returning a
				// kExhausted result
				result = PluginInterface::kExhausted;
			}
			break;

		case PluginInterface::kUnload:
			// clean up our act
			delete Settings::settings;
			result = PluginInterface::kAccepted;
			break;
	}
	return result;
}

const char *
PluginHelp(void)
{
	return "This is a sample Eddie plugin;\n"
	"An electric-C wannabe.\n";
}

const bool kDefaultOnState = false;
	// leave it off by default

Settings::Settings(PluginInterface *pluginInterface)
	:	on(kDefaultOnState)
{
	// register with Eddie to get called if our setting shows up in
	// the settings file
	pluginInterface->RegisterHandledShellCommand("EclecticCSetOn");
}

bool 
Settings::HandleSettingsCommand(PluginInterface *, const char ***argvp)
{
	// our registered setting was found while processing the settings file
	// here we get a chance to read it and set up ourselves accordingly
	const char **argv = *argvp;
	if (strcmp(argv[0], "EclecticCSetOn") == 0) {
		// we should always be here, unless we register more than one
		// setting
		if (strcasecmp(argv[1], "on") == 0)
			on = true;
		else if (strcasecmp(argv[1], "off") == 0)
			on = false;
		else
			return false;
			// wrong format
	}
	return true;
}

void 
Settings::Save(PluginInterface *pluginInterface)
{
	if (on != kDefaultOnState)
		pluginInterface->WriteSettings("%s %s\n", "EclecticCSetOn", on ? "on" : "off");

}

class BackgroundView : public BView {
	// a tiny utility class that we will use as our preference panel background
public:
	BackgroundView(BRect frame, const char *name, uint32 resizeMask, uint32 flags)
		:	BView(frame, name, resizeMask, flags)
		{}
		
	virtual void AttachedToWindow()
		{
			BView *parent = Parent();
			if (!parent)
				return;
			
			SetViewColor(parent->ViewColor());
			SetLowColor(parent->LowColor());
			SetHighColor(parent->HighColor());
		}
};

void 
Settings::AddPreferencePanel(PluginInterface *pluginInterface)
{
	BRect frame = pluginInterface->PrefsViewRect();
	frame.InsetBy(10, 10);
	frame.top += 3;
	
	// install a background view on which we will place all our
	// controls
	BView *view = new BackgroundView(frame, "background", B_WILL_DRAW, B_FOLLOW_ALL);

	// install all our controls;
	// er, just one check box really
	frame = view->Bounds();
	frame.OffsetBy(10, 10);
	frame.SetRightBottom(frame.LeftTop() + BPoint(100, 16));
	onCheckBox = new BCheckBox(frame, "on", "EclecticC on", 0);
	view->AddChild(onCheckBox);
	onCheckBox->SetValue(on);

	// add our preference panel using a SamplePrefsPanel proxy object
	SamplePrefsPanel *panel = new SamplePrefsPanel(view);
	panel->SetTarget(this);
	pluginInterface->AddAppPrefsPanel("Eclectic C Sample", panel);
	// the name used here will appear in the list of settings
	
	revertValue = on;
	// register the value of on at the time the panel is being built
	// so we can revert to it later
}

// revert/defaults/apply glue

void 
Settings::Apply()
{
	on = onCheckBox->Value();
}

void 
Settings::Revert()
{
	on = revertValue;
	onCheckBox->SetValue(revertValue);
}

void 
Settings::Defaults()
{
	on = kDefaultOnState;
	onCheckBox->SetValue(kDefaultOnState);
}

bool 
Settings::CanApply() const
{
	return on != onCheckBox->Value();
}

bool 
Settings::CanRevert() const
{
	return revertValue != onCheckBox->Value();
}

bool 
Settings::CanDefault() const
{
	return kDefaultOnState != onCheckBox->Value();
}

