/*
 * Chron
 * Copyright Jonathan Booth, Oct 13, 1996
 *
 * Last Update: Oct 13, 1996
 *
 * A rewrite of the X-windows clock 'Chron', except prettier.
 */

#include <time.h>
#include <Application.h>
#include <Bitmap.h>
#include "Clock.h"
#include "Color.h"

#define FF 0.4
#define SecFactor 0.45
#define MinFactor 0.3
#define HourFactor 0.2
#define delay 100000

		
Clock::Clock(BRect Area, color_space Mode)
	: BView(Area,"Clock",B_FOLLOW_ALL,B_WILL_DRAW|B_PULSE_NEEDED)
{
	// Set up the images
	Face = NULL;
	HandBit = new BBitmap(Area,Mode,TRUE);
	HandImg = new BView(Area,"Hands",0,0);

	// Setup the variables
	Center.Set((Area.right - Area.left) / 2.0,
			   (Area.bottom - Area.top) / 2.0);
	LenX = Area.right - Area.left;
	LenY = Area.bottom - Area.top;
	SecStat = TRUE;
	RedrawFace = TRUE;

	// Set up the past time
	OldSec=0;
}

Clock::~Clock()
{
	if (HandBit->Lock() && HandImg) {
		HandBit->RemoveChild(HandImg);
		HandBit->Unlock();
	}
	
	delete HandImg;
}

void Clock::AttachedToWindow()
{
	BView::AttachedToWindow();

	if (Window()->Lock()) {
		// Prepare to do what we need to
		BeginPicture(new BPicture);
		
		// Clear the drawing
		SetHighColor(Background);
		FillRect(Bounds());
		
		rgb_color FaceBackground;
		for (int x=0;x<30;x++) {
			FaceBackground.red = ((30-x)*255)/30;
			FaceBackground.green = FaceBackground.red;
			FaceBackground.blue = FaceBackground.red;

			SetHighColor(FaceBackground);
			FillArc(Center,LenX*FF,LenY*FF,(float)30+x,16.0);
			FillArc(Center,LenX*FF,LenY*FF,(float)90+x,16.0);
			FillArc(Center,LenX*FF,LenY*FF,(float)150+x,16.0);
			FillArc(Center,LenX*FF,LenY*FF,(float)210+x,16.0);
			FillArc(Center,LenX*FF,LenY*FF,(float)270+x,16.0);
			FillArc(Center,LenX*FF,LenY*FF,(float)330+x,16.0);
		}
		
		// While there is another color to do
		for (int y=0;y<360;y=y+60) {
			SetHighColor(FaceColors[((int)y/60)]);
			FillArc(Center,LenX*FF,LenY*FF,y,30);
			StrokeArc(Center,LenX*FF,LenY*FF,y+30,30);
		}

		
		// Got the picture?
		Face = EndPicture();

		Window()->Unlock();
	}
	
	// Pack drawing image into the bitmap
	if (HandImg) HandBit->AddChild(HandImg);
}

void Clock::DetachedFromWindow() {
	be_app->PostMessage(B_QUIT_REQUESTED);
}

void Clock::Draw(BRect updateRect)
{
	// Draw the image the *fast* way --
	//  draw it off of a bitmap, this causes
	//  zero flicker of the image. A very good thing.
	if (Window()->Lock()) {
		DrawBitmap(HandBit,updateRect,updateRect);
		Window()->Unlock();
	}
}

void Clock::UpdateImage()
{
	// Lotsa local vars here
	time_t t;
	tm *curTime;
	float SecArc,MinArc,HourArc;
	bool isam = TRUE;
	
	// Find out the time
	time(&t);
	curTime = localtime(&t);

	/*
	 * Have the seconds changed?
	 * This has a problem in theory if the bebox
	 * hung for a minute then resumed this task,
	 * however there is a bigger problem than that
	 * if the bebox does.
	 */
	if (((int)curTime->tm_sec) != OldSec) {
		// Update the clock updated time
		OldSec = curTime->tm_sec;
		
		// Ok, update the image if it exists
		if (HandBit->Lock()) {
			/*
			 * Calculate the arc-position of the
			 * various hands
			 */
			SecArc = (float)((curTime->tm_sec + 1) * 6);
			MinArc = (float)((curTime->tm_min * 6)
				 	  + (curTime->tm_sec / 10));
			HourArc = (float)(((curTime->tm_hour % 12) * 30)
					   + (curTime->tm_min / 2.0));
		
			/*
			 * Figure out if face image needs to
			 * redraw (pm = circle shrinking) or
			 * if we just past the hour and the
			 * circles dissapeared (am)
			 */
			if (curTime->tm_hour >=12) {
				isam = FALSE;
				RedrawFace = TRUE;
			}
			if (curTime->tm_sec == 0) {
				RedrawFace = TRUE;
			}
		
			/*
			 * Did we flag a clock face redraw?
			 */
			if (RedrawFace) {
				RedrawFace = FALSE;
				HandImg->DrawPicture(Face);
				HandImg->Sync();
			}
		
			// Second hand drawing
			if (SecStat) {
				HandImg->SetHighColor(SecHandColor);
				
				// Growing or shrinking circle
				if (isam) {
					HandImg->StrokeArc(Center,LenX*SecFactor,
								   LenY*SecFactor,
								   90,-SecArc);
				} else {
					HandImg->StrokeArc(Center,LenX*SecFactor,
								   LenY*SecFactor,
								   90,364-SecArc);
				}
			}
		
			// Minute hand
			HandImg->SetHighColor(MinHandColor);
			if (isam) {
				HandImg->FillArc(Center,LenX*MinFactor,
							   LenY*MinFactor,
							   90,0-MinArc);
			} else {
				HandImg->FillArc(Center,LenX*MinFactor,
							   LenY*MinFactor,
							   90,364-MinArc);
			}

			// Hour hand
			HandImg->SetHighColor(HourHandColor);
			if (isam) {
				HandImg->FillArc(Center,LenX*HourFactor,
							   LenY*HourFactor,
							   90,0-HourArc);
			} else {
				HandImg->FillArc(Center,LenX*HourFactor,
							   LenY*HourFactor,
							   90,364-HourArc);
			}

			/*
			 * Assure that the last drawing op
			 * completed before we redraw the
			 * image
			 */
			HandImg->Sync();
		
			/* 
			 * Must unlock what we locked
			 */
			HandBit->Unlock();
		}
		
		/*
		 * Force an update of the window
		 */
		if (Window() && Window()->Lock()) {
			Draw(Window()->Bounds());
			Window()->Unlock();
		}
	}
}

void Clock::ChangeSec()
{
	/*
	 * Set the seconds status to whatever we
	 * just registered as a change and flag
	 * a redraw of the background
	 */
	SecStat = !SecStat;
	RedrawFace = TRUE;
}

void Clock::Pulse() {
	UpdateImage();
	Draw(Window()->Bounds());
}