// BeCopy.cpp 990526
// 990612	type specified for dest files
// 990615	argv[1] caused seg violation, when argc=1
// 990616	stop locking dest files
// 990617	copy owner, group and permissions
// Copy a BFS file from the command line.
// It is equivalent to "cp" but it copies not only data but
// also attributes of a file.
//
// Taketora Yamagata, S I Electronics, Ltd.
//

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <storage/Directory.h>
#include <storage/Entry.h>
#include <storage/File.h>
#include <storage/Node.h>
#include <storage/Statable.h>
#include <storage/StorageDefs.h>
#include <storage/SymLink.h>
#include <kernel/fs_attr.h>
#include <support/SupportDefs.h>

void usage(void);
char* rindex(char*,char);
int copy(char*, char*);
int recursiveCopy(char*,char*);
void preserveTime(BNode*,BNode*);
void copyPermissions(BNode*,BNode*);

char	ansbuf[100];
char*	dataBuf;
#define DBSIZE 1000000

// Option variables.
bool backup = false;	// backup existing dest
bool linkAsLink = false;// preserve links
bool force = false;		// remove existing dest without asking
bool preserve = false;	// preserve creation date
bool recursive = false;	// recursive copy
bool update = false;	// copy only older or brand new files
bool verbose = false;	// explain what is being done

// get the rightmost key char in a string.
char*
rindex(char* nameStr, char key)
{
	char* sp = nameStr + strlen( nameStr );
	while (*--sp != key && sp >= nameStr ) ;
	if (sp < nameStr)
		return NULL;
	return sp;
}

void
usage(void)
{
	printf( "Usage: BeCopy [OPTION]... SOURCE DEST\n" );
	printf( "   or  BeCopy [OPTION]... SOURCE... DIRECTORY\n\n" );
	printf( "   -a  archive:    same as -dpr\n" );
	printf( "   -b  backup:     backup existing destinations.\n" );
	printf( "   -d  dereference:preserve links\n" );
	printf( "   -f  force:      remove existing destinations. no prompt\n" );
	printf( "   -h  help:       print this message\n" );
	printf( "   -p  preserve:   preserve creation time etc\n" );
	printf( "   -r  recursive:  copy recursively. If -d is not specified, symbolic\n" );
	printf( "                   links will be replaced by files they point\n" );
	printf( "   -u  update:     copy only older or brand new files\n" );
	printf( "   -v  verbose:    explain what is being done\n" );
	printf( "   --help          print this message\n" );
	exit(0);
}

// preserve creation and modification times
void
preserveTime(BNode *src, BNode *dst)
{
	time_t	creTime, modTime, accTime;
	src->GetCreationTime( &creTime );
	dst->SetCreationTime( creTime );
	src->GetModificationTime( &modTime );
	dst->SetModificationTime( modTime );
	src->GetAccessTime( &accTime );
	dst->SetAccessTime( accTime );
}

// copy owner, group and permissions
void
copyPermissions(BNode *src, BNode *dst)
{
	uid_t	owner;
	gid_t	group;
	mode_t	perms;
	src->GetOwner( &owner );
	dst->SetOwner( owner );
	src->GetGroup( &group );
	dst->SetGroup( group );
	src->GetPermissions( &perms );
	dst->SetPermissions( perms );
}

int
main( int32 argc, char *argv[] )
{
	// make sure we have enough arguments
	if (argc < 3) {
		if (argc<2 ||(strcmp(argv[1], "-h") && strcmp(argv[1], "--help")))//990615
			printf( "insufficient arguments\n" );
		usage( );
	}
		
	// get options	
	int32 opt;
	while ((opt = getopt( argc, argv, "abdfhpruv")) != -1) {
		switch(opt) {
		case 'a':
			linkAsLink = true;
			preserve = true;
			recursive = true;
			break;
		case 'b':
			backup = true;
			break;
		case 'd':
			linkAsLink = true;
			break;
		case 'f':
			force = true;
			break;
		case 'h':
			usage( );
			break;
		case 'p':
			preserve = true;
			break;
		case 'r':
			recursive = true;
			break;
		case 'u':
			update = true;
			break;
		case 'v':
			verbose = true;
			break;
		default:
			printf( "that option is not acceptable\n" );
			usage( );
			break;
		}
	}
	argc -= optind;
	argv += optind;
	if (argc < 2)
		usage( );
	
	struct stat stb;
	if (argc > 2 || recursive) {
		if (stat(argv[argc-1], &stb) < 0) {
			printf( "the last argument must be a directory\n" );
			usage( );
		}
		if (!S_ISDIR(stb.st_mode)) {
			printf( "the last argument must be a directory\n" );
			usage( );
		}
	}
	
	dataBuf = new char[DBSIZE];
	int rc = 0;
	for (int i = 0; i < argc -1; i++)
		rc |= copy( argv[i], argv[argc-1] );
	delete dataBuf;
	return rc;
}

int
copy( char* src, char* dst  )
{
	
	// open from file
	BNode srcNode( src );
	if (srcNode.InitCheck( ) != B_NO_ERROR) {
		printf( "BeCopy: cannot open %s\n", src );
		return 1;
	}
	struct stat stsrc, stdst;
	if (srcNode.GetStat( &stsrc ) != B_NO_ERROR ) {
		printf( "BeCopy: cannot get stat of %s\n", src );
		srcNode.Unset( );
		return 1;
	}
	
	char dstName[B_PATH_NAME_LENGTH];
	
	if (stat( dst, &stdst ) >= 0 && S_ISDIR( stdst.st_mode )) {
		char* last = rindex( src, '/' );
		if (last)	last++;
		else		last = src;
		if (strlen( dst ) + strlen( last ) >= B_PATH_NAME_LENGTH-1) {
			printf("BeCopy: %s/%s name too long\n", dst, last );
			srcNode.Unset( );
			return 1;
		}
		sprintf( dstName, "%s/%s", dst, last );
		dst = dstName;
	}
	
	if (recursive && S_ISDIR( stsrc.st_mode )) {
		if (stat( dst, &stdst ) < 0) {
			if (mkdir( dst, (int)stsrc.st_mode ) < 0) {
				printf( "BeCopy: cannot make directory %s\n", dst );
				srcNode.Unset( );
				return 1;
			}
			if (verbose)
				printf( "Directory %s is created\n", dst );
		} else if (!S_ISDIR( stdst.st_mode )) {
			printf( "BeCopy: %s is not a directory\n", dst );
			srcNode.Unset( );
			return 1;
		}
		recursiveCopy( src, dst );
		if (preserve) {
			BNode dstNode( dst );
			preserveTime( &srcNode, &dstNode );
			dstNode.Unset( );
		}
		srcNode.Unset( );
		return 0;
	}
	
	if (stat( dst, &stdst ) >= 0) {
		if (stsrc.st_dev==stdst.st_dev && stsrc.st_ino==stdst.st_ino){
			printf( "BeCopy: cannot copy %s to itself\n", src );
			srcNode.Unset( );
			return 1;
		}
		if (update)
			if (stsrc.st_mtime <= stdst.st_mtime) {
				if (verbose)
					printf( "%s is not newer,\n so %s is not replaced\n", src, dst );
				return 0;
			}
			goto checkBackup;
		if (!force && !backup)
			while (1) {
				printf( "%s exists. Overwrite it (y/n)?", dst );
				gets( ansbuf );
				if (ansbuf[0] == 'n') {
					srcNode.Unset( );
					return 1;
				}
				if (ansbuf[0] == 'y')
					break;
			}
		checkBackup:
		if (backup) {
			sprintf( dstName, "%s.%s", dst, "bak" );
			if (rename( dst, dstName ) < 0) {
				printf( "BeCopy: renaming from %s to %s failed\n", dst, dstName );
				return 1;
			}
			if (verbose)
				printf("%s is backed up\n", dst );
		} else {
			if (unlink( dst ) == -1) {
				if (S_ISDIR( stdst.st_mode )) {
					char comd[B_PATH_NAME_LENGTH + 10];
					sprintf( comd, "rm -r %s\n", dst );
					if (system( comd ) == -1 ) {
						printf( "BeCopy: %s cannot be removed\n", dst );
						return 1;
					}
				} else {
					printf( "BeCopy: %s cannot be removed\n", dst );
					return 1;
				}
			}
			if (verbose)
				printf( "%s is removed\n", dst );
		}
	}
	
	if (S_ISLNK( stsrc.st_mode )){
		char linkData[B_PATH_NAME_LENGTH];
		srcNode.Unset( );
		BSymLink srcLink( src );
		if (linkAsLink) {
			srcLink.ReadLink( linkData, B_PATH_NAME_LENGTH );
			char comd[B_PATH_NAME_LENGTH+50];
			sprintf( comd, "ln -s %s %s", linkData, dst );
			if (system( comd ) < 0) {
				printf("BeCopy: making a symlink %s at %s failed\n",
					linkData, dst );
				srcLink.Unset( );
				return 1;
			}
			if (preserve) {
				// preserve creation, modification times
				BNode dstLink( dst );
				preserveTime( (BNode *)&srcLink, &dstLink );
				dstLink.Unset( );
			}	
			if (verbose)
				printf( "Symlink %s -> %s created\n", dst, linkData );
			srcLink.Unset( );
			return 0;
		}
		if (verbose) {
			srcLink.ReadLink( linkData, B_PATH_NAME_LENGTH );
			printf( "Traverse %s\n", linkData );
		}
		srcLink.Unset( );
		goto fileCopy;	
	}

	if (S_ISREG( stsrc.st_mode )) {
		srcNode.Unset( );
	fileCopy:
		BFile srcFile( src, B_READ_ONLY );
		if ( srcFile.InitCheck( ) != B_NO_ERROR) {
			printf( "BeCopy: cannot open %s\n", src );
			return 1;
		}
		BFile dstFile( dst, B_WRITE_ONLY|B_CREATE_FILE|B_FAIL_IF_EXISTS );
		status_t status;
		if ((status=dstFile.InitCheck( )) != B_NO_ERROR) {
			printf( "BeCopy: %s cannot be created\n", dst );
			srcFile.Unset( );
			return 1;
		}
		// read src data and copy them to dst
		srcFile.Lock( );
		//dstFile.Lock( );												//990616
		int	n;
		for (;;) {
			n = srcFile.Read( dataBuf, DBSIZE );
			if (n == 0)
				break;
			if (n < 0) {
				printf( "BeCopy: %s read error\n", src );
				srcFile.Unlock( );
				//dstFile.Unlock( );									//990616
				return 1;
			}
			if (dstFile.Write( dataBuf, n ) != n) {
				printf( "BeCopy: %s write error\n", dst );
				srcFile.Unlock( );
				//dstFile.Unlock( );									//990616
				return 1;
			}
		}
		// read src attributes and copy them to dst
		char attrName[B_ATTR_NAME_LENGTH];
		while (srcFile.GetNextAttrName( attrName ) == B_NO_ERROR) {
			attr_info	srcAttrInfo;
			srcFile.GetAttrInfo( attrName, &srcAttrInfo );
			char *attrBuf = new char[srcAttrInfo.size];
			if (!attrBuf) {
				printf( "BeCopy: out of memory\n" );
				goto unlockExit;
			}
			ssize_t nSrc, nDst;
			nSrc = srcFile.ReadAttr( attrName, 0, 0, attrBuf, srcAttrInfo.size );
			if ( nSrc < 0 ) {
				printf( "cannot read attribute '%s'\n", attrName );
				delete attrBuf;
				goto unlockExit;
			}
			nDst = dstFile.WriteAttr( attrName, srcAttrInfo.type, 0, attrBuf, 
				srcAttrInfo.size );										//990612
			if ( nDst < 0 || nDst != nSrc ) {
				printf( " write attribute '%s' failed\n", attrName );
				delete attrBuf;
			unlockExit:
				srcFile.Unlock( );
				//dstFile.Unlock( );									//990616
				return 1;
			}
			delete attrBuf;
		}
		// copy owner, grouup, and permissions
		copyPermissions( (BNode*) &srcFile, (BNode*) &dstFile );		//990617
		// if '-p' option is set, preserve creation and modification times.
		if (preserve)
			preserveTime( (BNode*) &srcFile, (BNode*) &dstFile );
		srcFile.Unlock( );
		//dstFile.Unlock( );											//990616
		if (verbose)
			printf( "%s -> %s\n", src, dst );
	} else if (S_ISDIR( stsrc.st_mode )) {
		printf( "BeCopy: %s: omitting directory\n", src );
		return 1;
	}
	return 0;
}

int
recursiveCopy( char* src, char* dst )
{
	BDirectory srcDir( src );
	if (srcDir.InitCheck( ) != B_OK) {
		printf( "BeCopy: directory %s cannot open\n", src );
		return 1;
	}
	srcDir.Rewind( );
	char buf[4096];
	int32 count;
	dirent *dent;
	char srcName[B_PATH_NAME_LENGTH];
	while ((count = srcDir.GetNextDirents( (dirent*)buf, 4096 )) > 0) {
		dent = (dirent *)buf;
		while (count-- > 0) {
			if (dent->d_ino == 0)
				goto nextDent;
			if ( !strcmp(dent->d_name, "." )||!strcmp( dent->d_name, ".." ))
				goto nextDent;
			if (strlen(src)+2+strlen(dent->d_name) >= B_PATH_NAME_LENGTH){
				printf( "BeCopy: %s/%s Name too long\n", src, dent->d_name );
				goto nextDent;
			}
			sprintf( srcName, "%s/%s", src, dent->d_name );
			copy( srcName, dst );
		nextDent:
			dent = (dirent *)((char *)dent + dent->d_reclen);
		}
	}
	srcDir.Unset( );
	return 0;
}
