/******************************************************************************/
/* TestPlayer Player Interface.                                               */
/*                                                                            */
/* Written by Thomas Neumann.                                                 */
/******************************************************************************/
/* This is part of the APlayer Programming Package (APPP).                    */
/* Copyright (C) 1998-1999 by The APlayer-Team.                               */
/* All rights reserved.                                                       */
/*                                                                            */
/* This source, or parts thereof, may only be used in APlayer related         */
/* software. If you want to use it elsewhere, please contact the author for a */
/* permission.                                                                */
/******************************************************************************/


// PolyKit headers
#include "PString.h"
#include "PFile.h"

// Player headers
#include "TestPlayer.h"


/******************************************************************************/
/* Version                                                                    */
/******************************************************************************/
#define PlayerVersion		1.0



/******************************************************************************/
/* Tables                                                                     */
/******************************************************************************/
static uint32 freqs[] = {28160, 31584, 33472, 37600, 42176, 44704, 50176};

static int16 samp[256] =
{
	   0 << 8,    2 << 8,    4 << 8,    6 << 8,    8 << 8,   10 << 8,
	  12 << 8,   14 << 8,   16 << 8,   18 << 8,   20 << 8,   22 << 8,
	  24 << 8,   26 << 8,   28 << 8,   30 << 8,   32 << 8,   34 << 8,
	  36 << 8,   38 << 8,   40 << 8,   42 << 8,   44 << 8,   46 << 8,
	  48 << 8,   50 << 8,   52 << 8,   54 << 8,   56 << 8,   58 << 8,
	  60 << 8,   62 << 8,   64 << 8,   66 << 8,   68 << 8,   70 << 8,
	  72 << 8,   74 << 8,   76 << 8,   78 << 8,   80 << 8,   82 << 8,
	  84 << 8,   86 << 8,   88 << 8,   90 << 8,   92 << 8,   94 << 8,
	  96 << 8,   98 << 8,  100 << 8,  102 << 8,  104 << 8,  106 << 8,
	 108 << 8,  110 << 8,  112 << 8,  114 << 8,  118 << 8,  120 << 8,
	 122 << 8,  124 << 8,  126 << 8,  127 << 8,  126 << 8,  124 << 8,
	 122 << 8,  120 << 8,  118 << 8,  116 << 8,  114 << 8,  112 << 8,
	 110 << 8,  108 << 8,  106 << 8,  104 << 8,  102 << 8,  100 << 8,
	  98 << 8,   96 << 8,   94 << 8,   92 << 8,   90 << 8,   88 << 8,
	  86 << 8,   84 << 8,   82 << 8,   80 << 8,   78 << 8,   76 << 8,
	  74 << 8,   72 << 8,   70 << 8,   68 << 8,   66 << 8,   64 << 8,
	  62 << 8,   60 << 8,   58 << 8,   56 << 8,   54 << 8,   52 << 8,
	  50 << 8,   48 << 8,   46 << 8,   44 << 8,   42 << 8,   40 << 8,
	  38 << 8,   36 << 8,   34 << 8,   32 << 8,   30 << 8,   28 << 8,
	  26 << 8,   24 << 8,   22 << 8,   20 << 8,   18 << 8,   16 << 8,
	  14 << 8,   12 << 8,   10 << 8,    8 << 8,    6 << 8,    4 << 8,
	   2 << 8,    0 << 8, -  2 << 8, -  4 << 8, -  6 << 8, -  8 << 8,
	- 10 << 8, - 12 << 8, - 14 << 8, - 16 << 8, - 18 << 8, - 20 << 8,
	- 22 << 8, - 24 << 8, - 26 << 8, - 28 << 8, - 30 << 8, - 32 << 8,
	- 34 << 8, - 36 << 8, - 38 << 8, - 40 << 8, - 42 << 8, - 44 << 8,
	- 46 << 8, - 48 << 8, - 50 << 8, - 52 << 8, - 54 << 8, - 56 << 8,
	- 58 << 8, - 60 << 8, - 62 << 8, - 64 << 8, - 66 << 8, - 68 << 8,
	- 70 << 8, - 72 << 8, - 74 << 8, - 76 << 8, - 78 << 8, - 80 << 8,
	- 82 << 8, - 84 << 8, - 86 << 8, - 88 << 8, - 90 << 8, - 92 << 8,
	- 94 << 8, - 96 << 8, - 98 << 8, -100 << 8, -102 << 8, -104 << 8,
	-106 << 8, -108 << 8, -110 << 8, -112 << 8, -114 << 8, -116 << 8,
	-118 << 8, -120 << 8, -122 << 8, -124 << 8, -126 << 8, -127 << 8,
	-126 << 8, -124 << 8, -122 << 8, -120 << 8, -118 << 8, -116 << 8,
	-114 << 8, -112 << 8, -110 << 8, -108 << 8, -106 << 8, -104 << 8,
	-102 << 8, -100 << 8, - 98 << 8, - 96 << 8, - 94 << 8, - 92 << 8,
	- 90 << 8, - 88 << 8, - 86 << 8, - 84 << 8, - 82 << 8, - 80 << 8,
	- 78 << 8, - 76 << 8, - 74 << 8, - 72 << 8, - 70 << 8, - 68 << 8,
	- 66 << 8, - 64 << 8, - 62 << 8, - 60 << 8, - 58 << 8, - 56 << 8,
	- 54 << 8, - 52 << 8, - 50 << 8, - 48 << 8, - 46 << 8, - 44 << 8,
	- 42 << 8, - 40 << 8, - 38 << 8, - 36 << 8, - 34 << 8, - 32 << 8,
	- 30 << 8, - 28 << 8, - 26 << 8, - 24 << 8, - 22 << 8, - 20 << 8,
	- 18 << 8, - 16 << 8, - 14 << 8, - 12 << 8, - 10 << 8, -  8 << 8,
	-  6 << 8, -  4 << 8, -  2 << 8
};



/******************************************************************************/
/* Constructor                                                                */
/******************************************************************************/
TestPlayer::TestPlayer(APGlobalData *global) : APAddOnPlayer(global)
{
	printf("TestPlayer(): global = 0x%08X\n", (int)global);

	// Fill out the version we have compiled under
	addOnVersion  = AP_ADDON_VERSION;
	playerVersion = AP_PLAYER_VERSION;
	agentVersion  = AP_AGENT_VERSION;
}



/******************************************************************************/
/* Destructor                                                                 */
/******************************************************************************/
TestPlayer::~TestPlayer(void)
{
	printf("~TestPlayer()\n");
}



/******************************************************************************/
/* GetVersion() returns the version of the add-on.                            */
/*                                                                            */
/* Output: The add-on version.                                                */
/******************************************************************************/
float TestPlayer::GetVersion(void)
{
	printf("GetVersion()\n");
	return (PlayerVersion);
}



/******************************************************************************/
/* GetCount() returns the number of add-ons in the add-on.                    */
/*                                                                            */
/* Output: Is the number of the add-ons.                                      */
/******************************************************************************/
uint32 TestPlayer::GetCount(void)
{
	printf("GetCount()\n");
	return (1);
}



/******************************************************************************/
/* GetName() returns the name of the current add-on.                          */
/*                                                                            */
/* Input:  "index" is the add-on index starting from 0.                       */
/*                                                                            */
/* Output: The add-on name.                                                   */
/******************************************************************************/
PString TestPlayer::GetName(uint32 index)
{
	printf("GetName(): index = %ld\n", index);
	return ("TestPlayer");
}



/******************************************************************************/
/* GetDescription() returns the description of the current add-on.            */
/*                                                                            */
/* Input:  "index" is the add-on index starting from 0.                       */
/*                                                                            */
/* Output: The add-on description.                                            */
/******************************************************************************/
PString TestPlayer::GetDescription(uint32 index)
{
	printf("GetDescription(): index = %ld\n", index);
	return ("A little test player");
}



/******************************************************************************/
/* GetSupportFlags() returns some flags telling what the add-on supports.     */
/*                                                                            */
/* Output: Is the flags.                                                      */
/******************************************************************************/
uint32 TestPlayer::GetSupportFlags(void)
{
	printf("GetSupportFlags()\n");
	return (appSetPosition);
//	return (appSetPosition | appSamplePlayer);
}



/******************************************************************************/
/* GetModTypeString() returns the module type string for the current player.  */
/*                                                                            */
/* Input:  "index" is the player index number.                                */
/*                                                                            */
/* Output: The module type string.                                            */
/******************************************************************************/
PString TestPlayer::GetModTypeString(uint32 index)
{
	printf("GetModTypeString(): index = %ld\n", index);
	return ("");
}



/******************************************************************************/
/* ModuleCheck() tests the module to see if it's a Test module.               */
/*                                                                            */
/* Input:  "index" is the player index number.                                */
/*         "file" is a pointer to a APFile object with the file to check.     */
/*                                                                            */
/* Output: An APlayer result code.                                            */
/******************************************************************************/
ap_result TestPlayer::ModuleCheck(uint32 index, PFile *file)
{
	uint8 buf[19];

	printf("ModuleCheck(): index = %ld - file = 0x%08X\n", index, (int)file);

	// Check the module size
	if (file->GetLength() < 19)
		return (AP_UNKNOWN);

	// Check the mark
	file->SeekToBegin();
	file->Read(buf, 19);

	if (memcmp(buf, "APlayer test module", 19) == 0)
		return (AP_OK);

	return (AP_UNKNOWN);
}



/******************************************************************************/
/* LoadModule() will load the module into the memory.                         */
/*                                                                            */
/* Input:  "index" is the player index number.                                */
/*         "file" is a pointer to a file object with the file to check.       */
/*                                                                            */
/* Output: An APlayer result code.                                            */
/******************************************************************************/
ap_result TestPlayer::LoadModule(uint32 index, PFile *file)
{
	printf("LoadModule(): index = %ld - file = 0x%08X\n", index, (int)file);
	return (AP_OK);
}



/******************************************************************************/
/* ExtraLoad() will load all extra files needed.                              */
/*                                                                            */
/* Input:  "fileName" is the filename of the module with full path.           */
/*                                                                            */
/* Output: An APlayer result code.                                            */
/******************************************************************************/
ap_result TestPlayer::ExtraLoad(PString fileName)
{
	printf("ExtraLoad(): fileName = \"%s\"\n", (const char *)fileName);
	return (AP_OK);
}



/******************************************************************************/
/* InitPlayer() initialize the player.                                        */
/*                                                                            */
/* Output: True for success, false for an error.                              */
/******************************************************************************/
bool TestPlayer::InitPlayer(void)
{
	printf("InitPlayer()\n");
	return (true);
}



/******************************************************************************/
/* EndPlayer() ends the use of the player.                                    */
/******************************************************************************/
void TestPlayer::EndPlayer(void)
{
	printf("EndPlayer()\n");
}



/******************************************************************************/
/* InitSound() initialize the module to play.                                 */
/*                                                                            */
/* Input:  "songNum" is the subsong to play.                                  */
/******************************************************************************/
void TestPlayer::InitSound(uint16 songNum)
{
	printf("InitSound(): songNum = %d\n", songNum);

	writePlay = true;
	noteCount = 25;
	index     = 0;
	position  = 0;
}



/******************************************************************************/
/* EndSound() ends the current song.                                          */
/******************************************************************************/
void TestPlayer::EndSound(void)
{
	printf("EndSound()\n");
}



/******************************************************************************/
/* Play() is the main player function.                                        */
/******************************************************************************/
void TestPlayer::Play(void)
{
	if (writePlay)
	{
		printf("Play()\n");
		writePlay = false;
	}

	noteCount--;
	if (noteCount == 0)
	{
		noteCount = 25;

		virtChannels[0]->PlaySample(&samp, 0, 256, 16);
		virtChannels[0]->SetLoop(0, 256);
		virtChannels[0]->SetVolume(256);

		virtChannels[0]->SetFrequency(freqs[index++]);
		if (index == 7)
		{
			index = 0;
			position++;

			ChangePosition();

			if (position == 10)
			{
				position = 0;
				endReached = true;
			}
		}
	}
}



/******************************************************************************/
/* GetSamplePlayerInfo() will fill out the sample info structure given.       */
/*                                                                            */
/* Input:  "sampInfo" is a pointer to the sample info structure to fill.      */
/******************************************************************************/
void TestPlayer::GetSamplePlayerInfo(SamplePlayerInfo *sampInfo)
{
	printf("GetSamplePlayerInfo(): sampInfo = 0x%08X\n", (int)sampInfo);
}



/******************************************************************************/
/* GetModuleName() returns the name of the module.                            */
/*                                                                            */
/* Output: Is the module name.                                                */
/******************************************************************************/
PString TestPlayer::GetModuleName(void)
{
	printf("GetModuleName()\n");
	return ("");
}



/******************************************************************************/
/* GetAuthor() returns the name of the author.                                */
/*                                                                            */
/* Output: Is the author.                                                     */
/******************************************************************************/
PString TestPlayer::GetAuthor(void)
{
	printf("GetAuthor()\n");
	return ("");
}



/******************************************************************************/
/* GetVirtualChannels() returns the number of channels the module use.        */
/*                                                                            */
/* Output: Is the number of channels.                                         */
/******************************************************************************/
uint16 TestPlayer::GetVirtualChannels(void)
{
	printf("GetVirtualChannels()\n");
	return (2);
}



/******************************************************************************/
/* GetModuleChannels() returns the number of channels the module want to use. */
/*                                                                            */
/* Output: Is the number of required channels.                                */
/******************************************************************************/
uint16 TestPlayer::GetModuleChannels(void)
{
	printf("GetModuleChannels()\n");
	return (GetVirtualChannels());
}



/******************************************************************************/
/* GetSubSongs() returns the number of sub songs the module have.             */
/*                                                                            */
/* Output: Is a pointer to a subsong array.                                   */
/******************************************************************************/
uint16 *TestPlayer::GetSubSongs(void)
{
	printf("GetSubSongs()\n");

	subSongs[0] = 3;			// Number of subsongs
	subSongs[1] = 0;			// Start subsong number

	return (subSongs);
}



/******************************************************************************/
/* GetSongLength() returns the length of the song.                            */
/*                                                                            */
/* Output: The song length.                                                   */
/******************************************************************************/
int16 TestPlayer::GetSongLength(void)
{
	printf("GetSongLength()\n");
	return (10);
}



/******************************************************************************/
/* GetSongPosition() returns the current position in the song.                */
/*                                                                            */
/* Output: The song position.                                                 */
/******************************************************************************/
int16 TestPlayer::GetSongPosition(void)
{
	printf("GetSongPosition()\n");
	return (position);
}



/******************************************************************************/
/* SetSongPosition() sets a new position in the song.                         */
/*                                                                            */
/* Input:  "pos" is the new song position.                                    */
/******************************************************************************/
void TestPlayer::SetSongPosition(int16 pos)
{
	printf("SetSongPosition(): pos = %d\n", pos);

	position = pos;
	index    = 0;
}



/******************************************************************************/
/* GetInfoCount() returns the number of module info lines the player have.    */
/*                                                                            */
/* Output: The number of extra module info lines.                             */
/******************************************************************************/
uint32 TestPlayer::GetInfoCount(void)
{
	printf("GetInfoCount()\n");
	return (2);
}



/******************************************************************************/
/* GetInfoString() returns the description or value string on the line given. */
/*                                                                            */
/* Input:  "line" is the line starting from 0.                                */
/*         "type" is the type of string you want.                             */
/*                                                                            */
/* Output: The string you want.                                               */
/******************************************************************************/
PString TestPlayer::GetInfoString(uint32 line, ap_infoType type)
{
	printf("GetInfoString(): line = %ld - type = %d\n", line, type);

	PString retStr;

	if (type == apDescription)
	{
		// Description strings
		switch (line)
		{
			// Song Length
			case 0:
				retStr = "Song length";
				break;

			// Dummy
			case 1:
				retStr = "Dummy";
				break;
		}
	}
	else
	{
		// Value strings
		switch (line)
		{
			// Song Length
			case 0:
				retStr = "10";
				break;

			// Dummy
			case 1:
				retStr = "Just a dummy text";
				break;
		}
	}

	return (retStr);
}
