/* saferun - Runs a program but will SIGTERM (and then SIGKILL a few
 * seconds later) it if it hasn't finished within the specified time.
 *
 *
 * Written by Nick 'Zaf' Clifford <zaf@nrc.co.nz>
 * Copyright (C) 1999, Nick Clifford
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

#include <stdio.h>
#include <unistd.h>
#define _GNU_SOURCE
#include <getopt.h>
#include <string.h>
#include <stdlib.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <signal.h>

#define VERSION_MAJOR		0
#define VERSION_MINOR		5
#define VERSION_RELEASE		"beta"
#define DEFAULT_RETURN		120
#define DEFAULT_TIMEOUT_VAL	121

int timeout=0;
int failval=DEFAULT_RETURN;
int timeoutval=DEFAULT_TIMEOUT_VAL;
int verbose=0,silent = 0;


int child_pid;
int timed_out=0; /* Set to >0 if timed out */

void print_help();
void print_version();
int parse_options(int argc, char *argv[]);
void handle_alarm(int signum);

int main(int argc, char *argv[])
{
	int prog_idx;
	int status;
	prog_idx = parse_options(argc,argv);

	argv[argc] = NULL;
	if (verbose) {
		/* Print what we are executing */
		int i;
		fputs("executing: ",stdout);
		for (i = prog_idx; i < argc; i++) {
			fputs(argv[i],stdout);
			fputc(' ',stdout);
		}
		fputs("\n",stdout);
	}
	child_pid = fork();
	if (child_pid == -1) {
		if (!silent)
			fprintf(stderr,"fork() failed. error [%d:%s]\n",errno,strerror(errno));
		return failval;
	}
	if (child_pid == 0) {
		/* Child */
		if (execvp(argv[prog_idx], &(argv[prog_idx])) == -1) {
			if (!silent)
				fprintf(stderr,"execvp() failed. error [%d]: %s\n",errno,strerror(errno));
			return failval;
		}
		/* We, um.. shouldn't get here. ever */
		return 0;
	}
	/* Child is running as another process */

	/* Now setup the timeout */
	if (timeout) {
		signal(SIGALRM,handle_alarm);
		alarm(timeout);
	}
	
	if (waitpid(child_pid, &status, 0) == -1) {
		if (!silent)
			fprintf(stderr,"waitpid() failed. error [%d:%s]\n",errno,strerror(errno));
		alarm(0);
		return failval;
	}
	alarm(0);

	if (timed_out && timeoutval != -1) {
		/* The child timed out, and we are to return a code */
		if (verbose)
			printf("child exceeded time limit\n");
		return timeoutval;
	}
	/* Apparently, you can't use WIFEXITSTATUS unless WIFEXITED returns non zero
	 * but WIFEXITED appears to return 0 always
	 */
	/* if (WIFEXITED(status)) { */
		if (verbose)
			printf("child exited with error code %d\n",WEXITSTATUS(status));
		return (WEXITSTATUS(status));
	/* } */
	
}

/* When an alarm is raised, we first try and kill (-TERM) it, and raise another
 * alarm in 4 seconds, (so its possible it may take 4 seconds longer than
 * specified) on the 2nd alarm, we SIGKILL it. When the child dies, the waitpid
 * will return, and we will exit.
 */
int alarm_state = 0;

void handle_alarm(int signum)
{
	if (alarm_state == 0) {
		timed_out=1;
		if (verbose) 
			printf("Alarm reached.\n");
		/* Kill the child! */
		kill(child_pid,SIGTERM);
		alarm_state = 1;
		alarm(4);
	}
	else if (alarm_state == 1) {
		timed_out=2;
		if (verbose) 
			printf("Child still running?!.\n");
		/* Damnit, kill it for good! */
		kill(child_pid,SIGKILL);
		alarm_state = 2;
		exit(failval);
	}		
}

int parse_options(int argc, char *argv[])
{
	int r=0;
	struct option opts[]={
		{"version",no_argument,NULL,'V'},
		{"verbose",no_argument,NULL,'v'},
		{"silent",no_argument,NULL,'s'},
		{"failval",required_argument,NULL,'f'},
		{"rv_timeout",required_argument,NULL,'r'},
		{"timeout",required_argument,NULL,'t'},
		{"help",no_argument,NULL,'h'},
		{NULL,0,NULL,0}
	};

	while(r != -1) {
		r = getopt_long(argc,argv,"+Vvsrf:t:h",opts,NULL);
		switch (r) {
		case 's':
			silent = 1;
			verbose = 0;
			break;
		case 'v':
			verbose = 1;
			silent = 0;
			break;
		case 'f':
			failval = atoi(optarg);
			break;
		case 'r':
			timeoutval = atoi(optarg);
			break;
		case 't':
			timeout = atoi(optarg);
			break;
		case 'V':
			print_version();
			exit(0);
		case 'h':
			print_help();
			exit(0);
		}
	}
	/* optind contains the next arg # */
	if (optind >= argc) {
		fprintf(stderr,"Missing the program to execute!\n\n");
		print_help();
		exit(failval);
	}
	return optind;
}

void print_version()
{
	printf("saferun version %d.%d-%s\n", VERSION_MAJOR,
		VERSION_MINOR, VERSION_RELEASE);
	puts("Copyright (C) 1999, Nick 'Zaf' Clifford <zaf@nrc.co.nz>");
	puts("This program is licensed under the terms of the GNU General Public");
	puts("License version 2. For additional information on the license and");
	puts("copyright of this program, its source, and all other material;");
	puts("read the 'README' and 'Copying' file that  accompanies this programs ");
	puts("documentation.");
}

void print_help()
{
	puts("saferun [options] [program] [program args...]");
	puts("\nOptions include:");
	puts("  -t, --timeout <secs>    Kill the program if it hasn't finished in <secs>"); 
	puts("                          seconds.");
	puts("  -f, --failval <num>     If saferun fails for any reason, return this error");
	puts("                          code. saferun will fail if the program can't be ");
	puts("                          executed. (eg can't be found)");
	printf("                          (Default: %d)\n",DEFAULT_RETURN);
	puts("  -r, --rv_timeout <num>  If the timeout occurs, saferun returns this error");
	puts("                          code. Set to -1 to return the CHILDS RETURN VALUE");
	printf("                          (Default: %d)\n",DEFAULT_TIMEOUT_VAL);
	puts("  -v, --verbose           Verbose output. Just some status messages. (this");
	puts("                          option negates --silent)");
	puts("  -s, --silent            Silent output. Saferun Never displays any messages");
	puts("                          (even when an error occurs!) Normaly saferun is");
	puts("                          very quiet anyhow.");
	puts("  -V, --version           Display program version");
	puts("  -h, --help              Display help (this screen)");
	puts("");
	puts("[program] is the program to run. This may be the full path, relative path, or");
	puts("just the program name. If just the program name is specified, it is searched");
	puts("for in the PATH enviornment.");
	puts("[program args] are the command line arguments to send to the program that is");
	puts("to be run. These can be anything at all. Saferun ignores them.");
	puts("");
}
