// Robin Hood Web Server - A web server for BeOS
// Copyright (C) 1999 Joe Kloss

// This program is free software; you can redistribute it and/or 
// modify it under the terms of the GNU General Public License 
// as published by the Free Software Foundation; either version 2 
// of the License, or (at your option) any later version. 

// This program is distributed in the hope that it will be useful, 
// but WITHOUT ANY WARRANTY; without even the implied warranty of 
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the 
// GNU General Public License for more details. 

// You should have received a copy of the GNU General Public License 
// along with this program; if not, write to the Free Software 
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

// Contact Info:
// Author: Joe Kloss
// E-mail: axly@deltanet.com
// Postal Address: 25002 Ravenswood, Lake Forest, CA 92630, USA

#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <stdlib.h>
#include "StringUtils.h"
#include "ByteRanges.h"
#include "HTTPUtils.h"

ByteRangeSet::ByteRangeSet( void )
{
	
}

ByteRangeSet::ByteRangeSet( const char *setString )
{
	AddByteRange( setString );
}

ByteRangeSet::~ByteRangeSet( void )
{
	MakeEmpty();
}

void ByteRangeSet::AddByteRange( ByteRangeSpec *range )
{
	ByteRangeSpec		*newRange = new ByteRangeSpec;
	
	newRange->first = range->first;
	newRange->last = range->last;
	
	rangeSet.AddItem( newRange );
}

void ByteRangeSet::AddByteRange( int32 first, int32 last )
{
	ByteRangeSpec	range;
	
	range.first = first;
	range.last = last;
	AddByteRange( &range );
}

void ByteRangeSet::AddByteRange( const char *setString )
{
	const char		*sPtr = setString;
	ByteRangeSpec	range;
	char			token[64];
	const char		*tPtr;
	char			value[32];
	char			*eptr;
	
	// scan to beginning of byte range sets ( past "bytes=" if present )
	if( !(sPtr = strchr( sPtr, '=' )) )
		sPtr = setString;
	else
		sPtr++;
	// Get next byte range token until eof is reached
	while( *sPtr && (sPtr = get_next_token( token, sPtr, 64, "," )) )
	{
		tPtr = token;
		if( *token != '-' ) // is there a "first" value?
		{
			tPtr = get_next_token( value, tPtr, 32, "-" );
			range.first = strtol (value, &eptr, 10);
		}
		else
			range.first = -1;
		
		tPtr = get_next_token( value, tPtr, 32, "-" );
		if( *value != 0 )
		{
			range.last = strtol (value, &eptr, 10);
		}
		else
			range.last = -1;
		AddByteRange( &range );
	}
}

char *ByteRangeSet::ContentRangeString( char *buffer, int32 rangeIndex, int32 entityLength )
{
	if( rangeIndex >= CountRanges() )
		return NULL;
	
	FileOffsetSpec 	offset;
	
	GetFileOffset( &offset, entityLength, rangeIndex );
	sprintf( buffer, "bytes %ld-%ld/%ld", offset.offset, 
		offset.offset+offset.size-1, entityLength );
	
	return buffer;
}

char *ByteRangeSet::RangeString( char *buffer )
{
	char 			*sPtr = buffer;
	ByteRangeSpec	*range;
	
	sPtr += sprintf( sPtr, "bytes=" );
	for ( int32 i = 0; (range = GetRange(i)); i++ )
	{
		if( i != 0 )
			sPtr += sprintf( sPtr, "," );
		if( range->first != -1 )
			sPtr += sprintf( sPtr, "%ld", range->first );
		sPtr += sprintf( sPtr, "-" );
		if( range->last != -1 )
			sPtr += sprintf( sPtr, "%ld", range->last );
	}
	return buffer;
}


int32 ByteRangeSet::CountRanges( void )
{
	return rangeSet.CountItems();
}

ByteRangeSpec *ByteRangeSet::GetRange( int32 index )
{
	return (ByteRangeSpec *)rangeSet.ItemAt(index);
}

void ByteRangeSet::MakeEmpty()
{
	void *anItem;
	for ( int32 i = 0; (anItem = rangeSet.ItemAt(i)); i++ )
		delete anItem;
	rangeSet.MakeEmpty();
}

ByteRangeSpec *ByteRangeSet::GetFileOffset( FileOffsetSpec *offset, int32 fileSize, int32 rangeIndex )
{
	ByteRangeSpec	*range = GetRange( rangeIndex );
	
	if( !range )
		return NULL;
	
	if( range->first == -1 ) // Get last n bytes?
	{
		offset->offset = fileSize-range->last;
		offset->size = range->last;
		if( offset->offset < 0 )
		{
			offset->offset = 0;
			offset->size = fileSize;
		}
	}
	else if( range->last == -1 ) // Get bytes "first" to eof?
	{
		offset->offset = range->first;
		if( offset->offset > fileSize )
			offset->offset = 0;
		offset->size = fileSize - offset->offset;
	}
	else if( (range->first > range->last)||(range->first >= fileSize) ) // Error... get whole thing
	{
		offset->offset = 0;
		offset->size = fileSize;
	}
	else // get bytes "first" to "last"
	{
		offset->offset = range->first;
		offset->size = range->last - range->first + 1;
		if( offset->offset + offset->size > fileSize )
			offset->size = fileSize - offset->offset;
	}
	return range;
}

int32 ByteRangeSet::ContentLength( int32 entityLength )
{
	FileOffsetSpec 	offset;
	int32			totalBytes = 0;
	
	for ( int32 i = 0; (GetFileOffset( &offset, entityLength, i)); i++ )
		totalBytes += offset.size;
	return totalBytes;
}
