/*
 * ArpTelnet Copyright (c)1997-98 by Dianne Hackborn.
 * All rights reserved.
 *
 * Based (increasingly loosly) on WebTerm, a Web-based terminal
 * applet written in Java, which is
 * Copyright (C)1996 by the National Alliance for Computational
 * Science and Engineering (NACSE).
 * See <URL:http://www.nacse.org/> for more information.
 *
 * This code is not public domain, nor freely distributable.
 * Please direct any questions or requests to Dianne Hackborn,
 * at <hackbod@lucent.com> or <hackbod@angryredplanet.com>.
 *
 */

//#define DEBUG 1
//#define USE_STREAMS 1

#ifndef _AUTOLOCK_H
#include "support/Autolock.h"
#endif

#ifndef _APPLICATION_H
#include <app/Application.h>
#endif

#ifndef _SCROLL_VIEW_H
#include <interface/ScrollView.h>
#endif

#ifndef _MENU_BAR_H
#include <interface/MenuBar.h>
#endif

#ifndef _ALERT_H
#include <interface/Alert.h>
#endif

#ifndef ARPCOMMON_ARPDEBUG_H
#include <ArpCommon/ArpDebug.h>
#endif

#ifndef ARPCOMMON_ARPREMOTETERMINAL_H
#include <ArpCommon/ArpRemoteTerminal.h>
#endif

#ifndef ARPCOMMON_ARPEMULATOR_H
#include <ArpCommon/ArpEmulator.h>
#endif

#ifndef TERMINALWIN_H
#include "TerminalWin.h"
#endif

#ifndef CONNECTWIN_H
#include "ConnectWin.h"
#endif

#ifdef USE_LOOPBACK
#ifndef LOOPBACK_H
#include "Loopback.h"
#endif
#else
#ifndef ARPCOMMON_ARPTELNET_H
#include <ArpCommon/ArpTelnet.h>
#endif
#endif

#include <string.h>

// Some of these private messages are implemented, some aren't...
enum {
	QUERY_CONNECT_MSG = '.qcn',
	CONNECT_MSG = '.con',
	DISCONNECT_MSG = '.dcn',
	LOCK_SIZE_MSG = '.lsz',
	RESIZE_TO_FIT_MSG = '.rtf',
	SET_FONT_MSG = '.fnt',
	SET_FONT_SIZE_MSG = '.fsz',
	SET_HISTORY_USE_MSG = '.hus',
	SET_CRLF_MSG = '.scl',
	SET_SWAPBSDEL_MSG = '.sbd',
	SET_WRAP_MSG = '.swr',
	SET_INVERSE_MSG = '.sin',
	SET_PASTEQUERY_MSG = '.spq',
	SET_RMBPASTE_MSG = '.srp',
	SET_ISCROLL_MSG = '.sis',
	SET_OSCROLL_MSG = '.sos',
	SET_ENCODING_MSG = '.sen',
	SOFT_RESET_MSG = '.srs',
	HARD_RESET_MSG = '.hrs',
	ABOUT_EMULATION_MSG = '.aem',
	
	// Something old that we should probably just take out...
	TEST_MY_SCREEN = '.tst',
};

TerminalWin::TerminalWin(BPoint pos, const char* title,
							const char* abouttxt)
	: BWindow(BRect(pos.x,pos.y,pos.x+100,pos.y+100), title,
				B_DOCUMENT_WINDOW, 0),
	  manager(),
	  lockitem(NULL),
	  crlfitem(NULL), bsdelitem(NULL), wrapitem(NULL), inverseitem(NULL),
	  pqueryitem(NULL), rmbpasteitem(NULL), iscrollitem(NULL), oscrollitem(NULL),
	  terminal(NULL),
	  #ifdef USE_LOOPBACK
	  loopback(NULL)
	  #else
	  telnet(NULL)
	  #endif
{
	// set up a rectangle and instantiate a new view
	// view rect should be same size as window rect but with
	// left top at (0, 0)
	BRect aRect(Bounds());
	aRect.Set(0,0,
			  aRect.Width()-B_V_SCROLL_BAR_WIDTH,
			  aRect.Height()-B_H_SCROLL_BAR_HEIGHT);
	BMenuBar* menu = new BMenuBar(Bounds(),"MainMenu");
	BMenu* emumenu = NULL;
	BMenu* termmenu = NULL;
	if( menu ) {
		BMenu* file = new BMenu("File");
		if( file ) {
			BMenuItem *item; 
			
			item = new BMenuItem("Connect" B_UTF8_ELLIPSIS, 
	                             new BMessage(QUERY_CONNECT_MSG),
	                             'T'); 
			item->SetTarget(this); 
			file->AddItem(item);

			item = new BMenuItem("Disconnect", 
	                             new BMessage(DISCONNECT_MSG),0);
			item->SetTarget(this);
			file->AddItem(item);
			file->AddSeparatorItem();
			
			item = new BMenuItem(abouttxt,
	                             new BMessage(B_ABOUT_REQUESTED)); 
			item->SetTarget(be_app); 
			file->AddItem(item);
	
			item = new BMenuItem("Test Screen", 
	                             new BMessage(TEST_MY_SCREEN)); 
			item->SetTarget(this); 
			file->AddItem(item);
			file->AddSeparatorItem();
			
			item = new BMenuItem("Quit",
								new BMessage(B_QUIT_REQUESTED),'Q'); 
			item->SetTarget(be_app); 
			file->AddItem(item);
			
			menu->AddItem(file);
		}
		
		BMenu* edit = new BMenu("Edit");
		if( edit ) {
			BMenuItem *item; 
			
			item = new BMenuItem("Copy",
	                             new BMessage(B_COPY), 'C');
			item->SetTarget(NULL,this); 
			edit->AddItem(item);

			item = new BMenuItem("Paste", 
	                             new BMessage(B_PASTE), 'V');
			item->SetTarget(NULL,this); 
			edit->AddItem(item);
			
			item = new BMenuItem("Select All", 
	                             new BMessage(B_SELECT_ALL), 'A'); 
			item->SetTarget(NULL,this); 
			edit->AddItem(item);
			
			menu->AddItem(edit);
		}
		
		termmenu = new BMenu("Terminal");
		if( termmenu ) menu->AddItem(termmenu);
		
		emumenu = new BMenu("Emulation");
		if( emumenu ) menu->AddItem(emumenu);
		
		AddChild(menu);
		
		float w=0, h=0;
		menu->GetPreferredSize(&w,&h);
		aRect.top += h+1;
	}
	terminal = new ArpRemoteTerminal(aRect, "Test ArpTerminal");
	if( !terminal ) return;
	
	BScrollView		*scrview;
	scrview = new BScrollView("TerminalScrollView", terminal,
							  B_FOLLOW_ALL_SIDES, 0,
							  true, true, B_FANCY_BORDER);
	// add view to window
	if( scrview ) AddChild(scrview);
	else AddChild(terminal);

	DB(DBALL, cdb << "Creating remote device..." << endl);
	
	// Loopback() was the old remote device -- it just echoes
	// everything it receives.  But now that we (kind-of) have
	// a TELNET device...	
	BLooper* remote = NULL;
#ifdef USE_LOOPBACK
	loopback = new Loopback();
	if( !loopback ) return;
	remote = loopback;
#else
	telnet = new ArpTelnet();
	if( !telnet ) return;
	remote = telnet;
#endif
	
	DB(DBALL, cdb << "Adding remote device to terminal..." << endl);
	
	terminal->TermSetFixedSize(25,80);
	terminal->TermSetCursorPos(0,0);
	terminal->SetRemote(BMessenger(remote));
	
	DB(DBALL, cdb << "Locking my window." << endl);
	
	BAutolock wlock(this);
	terminal->MakeFocus(true);

	DB(DBALL, cdb << "Adding search paths to emulator manager..." << endl);
	
	// Tell the emulator manager which directories to look in.
	manager.AddEnvVar("ADDON_PATH","ArpTerminal");
	manager.AddDirectory(B_COMMON_ADDONS_DIRECTORY,"ArpTerminal");
	manager.AddDirectory(B_USER_ADDONS_DIRECTORY,"ArpTerminal");

	DB(DBALL, cdb << "Looking for emulations..." << endl);
	
	// Search for all available external add-ons, and add our
	// ArpEmulator class in case nothing else is there.
	manager.Start();
	manager.AddStaticEmulator(&ArpEmulator::AddonInterface);
	
	DB(DBALL, cdb << "Constructing menus..." << endl);
	
	// Construct 'Terminal' and 'Emulator' menus.
	if( terminal ) {
		DB(DBALL, cdb << "Setting terminal to use manager..." << endl);
		terminal->UseEmulatorManager(&manager);
		DB(DBALL, cdb << "Setting to XTERM emulator..." << endl);
		delete (terminal->SetEmulator("XTERM"));
		if( emumenu ) {
			DB(DBALL, cdb << "Creating emulator menu..." << endl);
			manager.CreateAddonMenu(emumenu);
			terminal->CheckEmulationMenu(emumenu);
			emumenu->SetTargetForItems(terminal);
			emumenu->AddSeparatorItem();
			BMenuItem* item = new BMenuItem("About Emulator", 
	                             new BMessage(ABOUT_EMULATION_MSG)); 
			item->SetTarget(NULL,this); 
			emumenu->AddItem(item);
		}
		if( termmenu ) {
			BMessage* msg;
			BMenuItem* item;
			
			DB(DBALL, cdb << "Creating terminal menu..." << endl);
			BMenu* winmenu = new BMenu("Usable Size");
			if( winmenu ) {
				termmenu->AddItem(winmenu);
				winmenu->SetRadioMode(true);
				msg = new BMessage(TERM_WINDOW_SIZE_MSG);
				if( msg ) {
					msg->AddInt32("rows",24);
					msg->AddInt32("columns",80);
					item = new BMenuItem("80 x 24", msg, '4');
					if( item ) {
						item->SetTarget(terminal);
						winmenu->AddItem(item);
					}
				}
				msg = new BMessage(TERM_WINDOW_SIZE_MSG);
				if( msg ) {
					msg->AddInt32("rows",25);
					msg->AddInt32("columns",80);
					item = new BMenuItem("80 x 25", msg, '5');
					if( item ) {
						item->SetTarget(terminal);
						winmenu->AddItem(item);
						item->SetMarked(true);
					}
				}
				msg = new BMessage(TERM_WINDOW_SIZE_MSG);
				if( msg ) {
					msg->AddInt32("rows",-1);
					msg->AddInt32("columns",80);
					item = new BMenuItem("80 Columns", msg, '8');
					if( item ) {
						item->SetTarget(terminal);
						winmenu->AddItem(item);
					}
				}
				msg = new BMessage(TERM_WINDOW_SIZE_MSG);
				if( msg ) {
					msg->AddInt32("rows",24);
					msg->AddInt32("columns",132);
					item = new BMenuItem("132 x 24", msg);
					if( item ) {
						item->SetTarget(terminal);
						winmenu->AddItem(item);
					}
				}
				msg = new BMessage(TERM_WINDOW_SIZE_MSG);
				if( msg ) {
					msg->AddInt32("rows",25);
					msg->AddInt32("columns",132);
					item = new BMenuItem("132 x 25", msg);
					if( item ) {
						item->SetTarget(terminal);
						winmenu->AddItem(item);
					}
				}
				msg = new BMessage(TERM_WINDOW_SIZE_MSG);
				if( msg ) {
					msg->AddInt32("rows",-1);
					msg->AddInt32("columns",132);
					item = new BMenuItem("132 Columns", msg);
					if( item ) {
						item->SetTarget(terminal);
						winmenu->AddItem(item);
					}
				}
				msg = new BMessage(LOCK_SIZE_MSG);
				if( msg ) {
					lockitem = new BMenuItem("Lock At ?", msg, 'L');
					if( lockitem ) {
						lockitem->SetTarget(this);
						winmenu->AddItem(lockitem);
					}
				}
				msg = new BMessage(TERM_WINDOW_SIZE_MSG);
				if( msg ) {
					msg->AddInt32("rows",-1);
					msg->AddInt32("columns",-1);
					item = new BMenuItem("Size With Window", msg, 'S');
					if( item ) {
						item->SetTarget(terminal);
						winmenu->AddItem(item);
					}
				}
			}
			
			BMenu* fontmenu = new BMenu("Font Family");
			if( fontmenu ) {
				termmenu->AddItem(fontmenu);
				winmenu->SetRadioMode(true);
				font_family cur_family;
				BFont font;
				if( terminal ) {
					terminal->GetFont(&font);
				}
				font.GetFamilyAndStyle(&cur_family,NULL);
				int32 num = count_font_families();
				for( int32 i=0; i<num; i++ ) {
					font_family name;
					uint32 flags = 0;
					if( get_font_family(i,&name,&flags) == B_NO_ERROR ) {
						if( flags&B_IS_FIXED ) {
							msg = new BMessage(SET_FONT_MSG);
							if( msg ) {
								msg->AddString("family",&name[0]);
								item = new BMenuItem(&name[0], msg);
								if( item ) {
									item->SetTarget(this);
									fontmenu->AddItem(item);
									if( !strcmp(name,cur_family) ) {
										item->SetMarked(true);
									}
								}
							}
						}
					}
				}
			}
			
			BMenu* fsizemenu = new BMenu("Font Size");
			if( fsizemenu ) {
				termmenu->AddItem(fsizemenu);
				fsizemenu->SetRadioMode(true);
				BFont font;
				if( terminal ) {
					terminal->GetFont(&font);
				}
				static const int32 sizes[] = {
					1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14,
					16, 18, 20, 24, 28, 32, 40, 60, 80, 120, 0 };
				for( int32 i=0; sizes[i]; i++ ) {
					msg = new BMessage(SET_FONT_SIZE_MSG);
					if( msg ) {
						msg->AddInt32("size",sizes[i]);
						char buffer[32];
						sprintf(&buffer[0],"%d",sizes[i]);
						item = new BMenuItem(&buffer[0], msg);
						if( item ) {
							item->SetTarget(this);
							fsizemenu->AddItem(item);
							if( sizes[i] == font.Size() ) {
								item->SetMarked(true);
							}
						}
					}
				}
			}
			
			BMenu* encmenu = new BMenu("Encoding");
			if( encmenu ) {
				termmenu->AddItem(encmenu);
				encmenu->SetRadioMode(true);
				static const char* encnames[] = {
					"ISO 8859 Latin 1",
					"ISO 8859 Latin 2",
					"ISO 8859 Latin 3",
					"ISO 8859 Latin 4",
					"ISO 8859 Latin/Cyrillic",
					"ISO 8859 Latin/Arabic",
					"ISO 8859 Latin/Hebrew",
					"ISO 8859 Latin 5",
					"ISO 8859 Latin 6",
					"Macintosh Roman",
					NULL
				};
				static int32 enccodes[] = {
					B_ISO_8859_1,
					B_ISO_8859_2,
					B_ISO_8859_3,
					B_ISO_8859_4,
					B_ISO_8859_5,
					B_ISO_8859_6,
					B_ISO_8859_7,
					B_ISO_8859_8,
					B_ISO_8859_9,
					B_ISO_8859_10,
					B_MACINTOSH_ROMAN 
				};
				for( int32 i=0; encnames[i] != NULL; i++ ) {
					msg = new BMessage(SET_ENCODING_MSG);
					if( msg ) {
						msg->AddInt32("encoding",enccodes[i]);
						item = new BMenuItem(encnames[i], msg);
						if( item ) {
							item->SetTarget(this);
							encmenu->AddItem(item);
							if( enccodes[i] == terminal->TermEncoding() ) {
								item->SetMarked(true);
							}
						}
					}
				}
			}
			
			BMenu* histmenu = new BMenu("History Use");
			if( histmenu ) {
				termmenu->AddItem(histmenu);
				histmenu->SetRadioMode(true);
				int32 curuse = 0;
				if( terminal ) curuse = terminal->HistoryUse();
				msg = new BMessage(SET_HISTORY_USE_MSG);
				if( msg ) {
					msg->AddInt32("value",ArpCoreTerminal::HISTORY_NONE);
					item = new BMenuItem("Off", msg, 'O');
					if( item ) {
						item->SetTarget(this);
						histmenu->AddItem(item);
						if( curuse == msg->FindInt32("value") ) {
							item->SetMarked(true);
						}
					}
				}
				msg = new BMessage(SET_HISTORY_USE_MSG);
				if( msg ) {
					msg->AddInt32("value",ArpCoreTerminal::HISTORY_MINIMUM);
					item = new BMenuItem("Minimum", msg, 'M');
					if( item ) {
						item->SetTarget(this);
						histmenu->AddItem(item);
						if( curuse == msg->FindInt32("value") ) {
							item->SetMarked(true);
						}
					}
				}
				msg = new BMessage(SET_HISTORY_USE_MSG);
				if( msg ) {
					msg->AddInt32("value",ArpCoreTerminal::HISTORY_MODERATE);
					item = new BMenuItem("Normal", msg, 'N');
					if( item ) {
						item->SetTarget(this);
						histmenu->AddItem(item);
						if( curuse == msg->FindInt32("value") ) {
							item->SetMarked(true);
						}
					}
				}
				msg = new BMessage(SET_HISTORY_USE_MSG);
				if( msg ) {
					msg->AddInt32("value",ArpCoreTerminal::HISTORY_AGGRESSIVE);
					item = new BMenuItem("Aggressive", msg, 'G');
					if( item ) {
						item->SetTarget(this);
						histmenu->AddItem(item);
						if( curuse == msg->FindInt32("value") ) {
							item->SetMarked(true);
						}
					}
				}
			}
			
			BMenu* modemenu = new BMenu("Operation Modes");
			if( modemenu ) {
				termmenu->AddItem(modemenu);
				crlfitem = new BMenuItem("LF -> CRLF",
								new BMessage(SET_CRLF_MSG),'F');
				if( crlfitem ) {
					crlfitem->SetTarget(this);
					modemenu->AddItem(crlfitem);
				}
				bsdelitem = new BMenuItem("Swap Backspace/Del",
								new BMessage(SET_SWAPBSDEL_MSG),'B');
				if( bsdelitem ) {
					bsdelitem->SetTarget(this);
					modemenu->AddItem(bsdelitem);
				}
				wrapitem = new BMenuItem("Wrap Cursor",
								new BMessage(SET_WRAP_MSG),'P');
				if( wrapitem ) {
					wrapitem->SetTarget(this);
					modemenu->AddItem(wrapitem);
				}
				inverseitem = new BMenuItem("Inverse",
								new BMessage(SET_INVERSE_MSG),'I');
				if( inverseitem ) {
					inverseitem->SetTarget(this);
					modemenu->AddItem(inverseitem);
				}
				
				modemenu->AddSeparatorItem();
			
				pqueryitem = new BMenuItem("Verify Paste",
								new BMessage(SET_PASTEQUERY_MSG),0);
				if( pqueryitem ) {
					pqueryitem->SetTarget(this);
					modemenu->AddItem(pqueryitem);
				}
				rmbpasteitem = new BMenuItem("Quick Paste",
								new BMessage(SET_RMBPASTE_MSG),0);
				if( rmbpasteitem ) {
					rmbpasteitem->SetTarget(this);
					modemenu->AddItem(rmbpasteitem);
				}
				iscrollitem = new BMenuItem("Scroll on Input",
								new BMessage(SET_ISCROLL_MSG),0);
				if( iscrollitem ) {
					iscrollitem->SetTarget(this);
					modemenu->AddItem(iscrollitem);
				}
				oscrollitem = new BMenuItem("Scroll on Output",
								new BMessage(SET_OSCROLL_MSG),0);
				if( oscrollitem ) {
					oscrollitem->SetTarget(this);
					modemenu->AddItem(oscrollitem);
				}
				update_modes();
			}

			termmenu->AddSeparatorItem();
			
			item = new BMenuItem("Soft Reset", 
	                             new BMessage(SOFT_RESET_MSG),'R'); 
			if( item ) {
				item->SetTarget(this);
				termmenu->AddItem(item);
			}
			
			item = new BMenuItem("Hard Reset", 
	                             new BMessage(HARD_RESET_MSG),'H'); 
			if( item ) {
				item->SetTarget(this);
				termmenu->AddItem(item);
			}
		
			item = new BMenuItem("Resize To Fit", 
	                             new BMessage(RESIZE_TO_FIT_MSG),'Y'); 
			if( item ) {
				item->SetTarget(this);
				termmenu->AddItem(item);
			}
		}
	}
	
	// An unsuccessful attempt to dispatch cursor keys with
	// command down to the terminal...
#if 0
	BMessage* msg = new BMessage(B_KEY_DOWN);
	if( msg ) {
		msg->AddInt32("modifiers",B_COMMAND_KEY|B_OPTION_KEY);
		msg->AddInt32("raw_char",B_UP_ARROW);
		msg->AddInt32("byte",B_UP_ARROW);
		msg->AddString("bytes","");
		AddShortcut(B_UP_ARROW,0,msg,terminal);
	}
#endif

	DB(DBALL, cdb << "Setting window size..." << endl);
	// Snap window to correct size and place cursor at top/left.
	ResizeToTerm();
	DB(DBALL, cdb << "Done constructing terminal window!" << endl);
}

TerminalWin::~TerminalWin()
{
	#ifdef USE_LOOPBACK
	if( loopback ) {
		loopback->Lock();
		loopback->Quit();
	}
	#else
	if( telnet ) {
		telnet->Lock();
		telnet->Quit();
	}
	#endif
}

void TerminalWin::update_modes(void)
{
	if( terminal ) {
		int32 mode = terminal->TermMode();
		if( crlfitem )
			crlfitem->SetMarked(mode&terminal->TERM_MODENEWLINE);
		if( bsdelitem )
			bsdelitem->SetMarked(mode&terminal->TERM_MODESWAPBSDEL);
		if( wrapitem )
			wrapitem->SetMarked(!(mode&terminal->TERM_MODENOWRAP));
		if( inverseitem )
			inverseitem->SetMarked(mode&terminal->TERM_MODEINVERSE);
		if( pqueryitem )
			pqueryitem->SetMarked(terminal->PasteVerified());
		if( rmbpasteitem )
			rmbpasteitem->SetMarked(terminal->RMBPaste());
		if( iscrollitem )
			iscrollitem->SetMarked(((int)terminal->AutoScrollMode())
									& terminal->AUTOSCROLL_INPUT);
		if( oscrollitem )
			oscrollitem->SetMarked(((int)terminal->AutoScrollMode())
									& terminal->AUTOSCROLL_OUTPUT);
	}
}

// Start the TELNET session.
void TerminalWin::Connect(const char* rhost, int32 rport)
{
	#ifndef USE_LOOPBACK
	if(telnet) telnet->Connect(rhost,rport);
	#endif
}

void TerminalWin::PrintToTerm(ArpString& str, int32 flags)
{
	if( terminal ) terminal->TermSendTTY((const ichar*)str,
											str.Length(),flags);
}

void TerminalWin::Zoom(BPoint leftTop, float width, float height)
{
	ResizeToTerm();
}

void TerminalWin::ResizeToTerm(int32 rows, int32 cols)
{
	if( terminal ) {
		terminal->TermGetSize(rows < 0 ? &rows:NULL,
								cols < 0 ? &cols:NULL);
		float w,h;
		terminal->GetNeededSize(cols,rows,&w,&h);
		BRect win = Frame();
		BRect inr = terminal->Frame();
		w = w + floor(win.Width()-inr.Width())-1;
		h = h + floor(win.Height()-inr.Height())-1;
		ResizeTo(w,h);
		// For some reason, the terminal is not getting updated
		// corrently when we manually set the window size...
		// (It's only drawing the newly exposed regions.)
		// This is a temporary hack to fix this.
		terminal->Invalidate();
	}
}

void TerminalWin::DispatchMessage(BMessage *message,
								  BHandler *handler)
{
	DB(DBALL,if( message->what != B_MOUSE_MOVED )
				cerr << "Dispatch message to " << handler
					<< ": " << *message << endl);
	if( message && message->what == B_KEY_DOWN || message->what == B_KEY_UP ) {
		BView* view = dynamic_cast<BView*>(handler);
		if( view ) {
			// HACK HACK HACK...  when using non-unicode encodings, CTRL-space doesn't
			// seem to get delivered to the view...  so force it to get there.
			int32 mods = 0;
			int32 key = 0;
			message->FindInt32("modifiers",&mods);
			message->FindInt32("raw_char",&key);
			if( (key == ' ' || key == '2') && (mods&B_CONTROL_KEY) != 0 ) {
				char byte = 0;
				if( message->what == B_KEY_DOWN ) view->KeyDown(&byte,1);
				else view->KeyUp(&byte,1);
				return;
			}
		}
	}
	inherited::DispatchMessage(message,handler);
}

void TerminalWin::MessageReceived(BMessage* message)
{
	if( !message ) return;
	
	switch( message->what ) {
		case B_PASTE: {
			// Just used for my own testing purposes...
			if(  be_clipboard->Lock() ) {
				BMessage* clip = be_clipboard->Data();
				DB(DBALL, cerr << "Clipboard data: " << *clip
						<< endl);
				be_clipboard->Unlock();
			}
			inherited::MessageReceived(message);
		} break;
		case QUERY_CONNECT_MSG: {
			#ifndef USE_LOOPBACK
			if( telnet ) {
				BRect frm = Frame();
				ConnectWin* cn = new ConnectWin(frm,this,
											telnet->Host(),
											telnet->Port());
				if( cn ) {
					BRect cfrm = cn->Frame();
					cn->MoveTo( frm.left
									+ (frm.Width()-cfrm.Width())/2,
								 frm.top
								 	+ (frm.Height()-cfrm.Height())/2);
					cn->Show();
				}
			}
			#endif
		} break;
		case CONNECT_SELECTED_MSG: {
			const char* host = NULL;
			int32 port = 23;
			message->FindString("host",&host);
			message->FindInt32("port",&port);
			Connect(host,port);
		} break;
		case DISCONNECT_MSG: {
			#ifndef USE_LOOPBACK
			if( telnet ) telnet->Disconnect();
			#endif
		} break;
		case RESIZE_TO_FIT_MSG: {
			ResizeToTerm();
		} break;
		case LOCK_SIZE_MSG: {
			int32 rows=0, cols=0;
			if( message->FindInt32("rows",&rows) == B_NO_ERROR
				&& message->FindInt32("columns",&cols) == B_NO_ERROR ) {
				if( rows > 0 && cols > 0 && terminal ) {
					terminal->TermSetFixedSize(rows,cols);
				}
			}
		} break;
		case SET_FONT_MSG: {
			const char* name;
			if( message->FindString("family",&name) == B_NO_ERROR ) {
				if( name && terminal ) {
					BFont font;
					font_family family;
					strcpy(&family[0],name);
					int32 rows, cols;
					terminal->TermGetViewSize(&rows,&cols);
					terminal->GetFont(&font);
					font.SetFamilyAndStyle(family,NULL);
					terminal->SetFont(&font);
					ResizeToTerm(rows,cols);
				}
			}
		} break;
		case SET_FONT_SIZE_MSG: {
			int32 size = 0;
			if( message->FindInt32("size",&size) == B_NO_ERROR ) {
				if( terminal ) {
					BFont font;
					int32 rows=0, cols=0;
					terminal->TermGetViewSize(&rows,&cols);
					terminal->GetFont(&font);
					font.SetSize(size);
					terminal->SetFont(&font);
					ResizeToTerm(rows,cols);
				}
			}
		} break;
		case SET_HISTORY_USE_MSG: {
			int32 value;
			if( message->FindInt32("value",&value) == B_NO_ERROR ) {
				if( terminal ) {
					terminal->SetHistoryUse(
						(ArpCoreTerminal::HistoryUseType)value);
				}
			}
		} break;
		case SET_CRLF_MSG: {
			if( terminal ) {
				terminal->TermSetMode( terminal->TermMode()
									^ terminal->TERM_MODENEWLINE);
				update_modes();
			}
		} break;
		case SET_SWAPBSDEL_MSG: {
			if( terminal ) {
				terminal->TermSetMode( terminal->TermMode()
									^ terminal->TERM_MODESWAPBSDEL);
				update_modes();
			}
		} break;
		case SET_WRAP_MSG: {
			if( terminal ) {
				terminal->TermSetMode( terminal->TermMode()
									^ terminal->TERM_MODENOWRAP);
				update_modes();
			}
		} break;
		case SET_INVERSE_MSG: {
			if( terminal ) {
				terminal->TermSetMode( terminal->TermMode()
									^ terminal->TERM_MODEINVERSE);
				update_modes();
				terminal->TermClean();
			}
		} break;
		case SET_PASTEQUERY_MSG: {
			if( terminal ) {
				terminal->SetPasteVerified(!terminal->PasteVerified());
				update_modes();
			}
		} break;
		case SET_RMBPASTE_MSG: {
			if( terminal ) {
				terminal->SetRMBPaste(!terminal->RMBPaste());
				update_modes();
			}
		} break;
		case SET_ISCROLL_MSG: {
			if( terminal ) {
				terminal->SetAutoScrollMode(
					(ArpTerminal::AutoScrollType)
					(((int)terminal->AutoScrollMode())
						^ terminal->AUTOSCROLL_INPUT) );
				update_modes();
			}
		} break;
		case SET_OSCROLL_MSG: {
			if( terminal ) {
				terminal->SetAutoScrollMode(
					(ArpTerminal::AutoScrollType)
					(((int)terminal->AutoScrollMode())
						^ terminal->AUTOSCROLL_OUTPUT) );
				update_modes();
			}
		} break;
		case SET_ENCODING_MSG: {
			int32 value;
			if( message->FindInt32("encoding",&value) == B_NO_ERROR ) {
				if( terminal ) {
					terminal->TermSetEncoding(value);
				}
			}
		} break;
		case SOFT_RESET_MSG: {
			if( terminal ) {
				terminal->Emulator().Reset(false);
				terminal->TermClean();
			}
		} break;
		case HARD_RESET_MSG: {
			if( terminal ) {
				terminal->Emulator().Reset(true);
				terminal->TermClean();
			}
		} break;
		case ABOUT_EMULATION_MSG: {
			const int32 bsize = 8192;
			char buffer[bsize];
			buffer[0] = 0;
			if( terminal ) {
				const ArpEmulationType* type =
					terminal->Emulator().EmulationType();
				if( type ) {
					strncat(&buffer[0], "Emulation name: ", bsize);
					strncat(&buffer[0], type->Name, bsize);
					if( type->Synonym ) {
						strncat(&buffer[0], "\nOfficial name: ", bsize);
						strncat(&buffer[0], type->Synonym, bsize);
					}
					strncat(&buffer[0], "\nLong name: ", bsize);
					strncat(&buffer[0], type->LongName, bsize);
					const ArpEmulatorAddon* addon =
						manager.EmulatorAddonForType(type->Name);
					if( addon ) {
						strncat(&buffer[0], "\n\nAddon name: ", bsize);
						strncat(&buffer[0], addon->Name, bsize);
						strncat(&buffer[0], "\nCompany: ", bsize);
						strncat(&buffer[0], addon->Company, bsize);
						strncat(&buffer[0], "\nCopyright: ", bsize);
						strncat(&buffer[0], addon->Copyright, bsize);
						strncat(&buffer[0], "\nBuild Date/Time: ", bsize);
						strncat(&buffer[0], addon->BuildDate, bsize);
						strncat(&buffer[0], " / ", bsize);
						strncat(&buffer[0], addon->BuildTime, bsize);
						strncat(&buffer[0], "\nVersion: ", bsize);
						sprintf(&buffer[strlen(&buffer[0])],"%d",
									addon->Version);
						strncat(&buffer[0], "\n\n", bsize);
						strncat(&buffer[0], addon->Description, bsize);
					}
				}
			}
			if( !buffer[0] ) strcat(&buffer[0],
									"No information available.");
			BAlert* alert = new BAlert("About Emulator",&buffer[0],
											"Dismiss");
			if( alert ) {
				alert->Go(NULL);
			}
		} break;
		case TEST_MY_SCREEN: {
			if( terminal ) {
				ArpTerminal& t = *terminal;
				t.TermSetStyle(t.TERM_STYLEPLAIN);
				PrintToTerm(ArpString("\r\nExample ArpTerminal Rendering\r\n"));
				PrintToTerm(ArpString("\r\nCharacter styles: "));
				t.TermSetStyle(t.TERM_STYLEPLAIN);
				PrintToTerm(ArpString("Plain "));
				t.TermSetStyle(t.TERM_STYLEINVERSE);
				PrintToTerm(ArpString("Inverse "));
				t.TermSetStyle(t.TERM_STYLEBOLD);
				PrintToTerm(ArpString("Bold "));
				t.TermSetStyle(t.TERM_STYLEITALIC);
				PrintToTerm(ArpString("Italic "));
				t.TermSetStyle(t.TERM_STYLEUNDERSCORE);
				PrintToTerm(ArpString("Underscore "));
				t.TermSetStyle(t.TERM_STYLEINVERSE|t.TERM_STYLEBOLD
						|t.TERM_STYLEITALIC|t.TERM_STYLEUNDERSCORE);
				PrintToTerm(ArpString("Combination"));
				t.TermSetStyle(t.TERM_STYLEPLAIN);
				PrintToTerm(ArpString("\r\n\r\nColors:\r\n"));
				int i, j;
//				for( i=0; i<t.TERM_NUMTEXTCOLORS; i++ ) {
//					for( j=0; j<t.TERM_NUMTEXTCOLORS; j++ ) {
				for( i=0; i<9; i++ ) {
					for( j=0; j<9; j++ ) {
						t.TermSetBackColor(i);
						t.TermSetForeColor(j);
						//printf("\033[3%dm\033[4%dmCOLOR ",j,i);
						PrintToTerm(ArpString("COLOR "),t.TERM_OUTPARTIAL);
					}
					t.TermSetBackColor(0);
					t.TermSetForeColor(0);
					//printf("\033[0m\r\n");
					PrintToTerm(ArpString("\r\n"));
				}
					
			}
		} break;
		default:
			inherited::MessageReceived(message);
	}
}

void TerminalWin::MenusBeginning()
{
	if( terminal && lockitem && !lockitem->IsMarked() ) {
		BMessage* msg = new BMessage(LOCK_SIZE_MSG);
		if( msg ) {
			int32 rows=0, cols=0;
			terminal->TermGetViewSize(&rows,&cols);
			char labtext[128];
			sprintf(&labtext[0],"Lock At %d x %d",rows,cols);
			lockitem->SetLabel(&labtext[0]);
			msg->AddInt32("rows",rows);
			msg->AddInt32("columns",cols);
			lockitem->SetMessage(msg);
		}
	}
	update_modes();
}

bool TerminalWin::QuitRequested()
{
	be_app->PostMessage(B_QUIT_REQUESTED);
	return(true);
}
