/* -------------------------------------------------------------------

	Title:
		WMPolyFit

	Description:
		Polymomial fit routine using Singular Value Decomposition (SVD).  
		Adapted from "Numerical Recipes" by W.H.Press.  Adaptation from    
		Pascal

	Author:
		Stephen C. Wardlaw, M.D.
		Yale University School Of Medicine
		462-CB  Y-NHH
		20 York St.
		New Haven, CT  06504

	Edit History:
		27 December, 1997
			Adapted for BeOS
		20 Jan, 94
			Re-written for MetroWerks C++
		19 Mar, 92
			Recompiled under MW Vers 4.0
		08 Aug, 91
			Adapted to Macintosh model
		28 Mar, 89
			First IBM version

----------------------------------------------------------------------

	Finding a linear polynomial fit consists of finding the coefficents
   Ai of the polynomial
      P(x) = A0*X0(x) + A2*X2(x) +...+ Ai*Xi(x) +...+ An*Xn(x)
   The Xi are themselves polynomials, and are called the "basis"
   polynomials.  In the simplest and most familiar case, these basis
   polynomials are just the powers of x. (Xi = x**i, the ith power of x).
      P(x) = A0*(x**0) + A2*(x**2) +...+ Ai*(x**i) +...+ An*(x**n)
   More complicated basis polynomials can be used to spread the fitting
   error more evenly.

      The method used, Singular Value Decomposition, is a little slower
   than gaussian methods, but it is more reliable in cases where
   gaussian methods might get hung up on round-off error

----------------------------------------------------------------------
*  Method Usage
----------------------------------------------------------------------

	Initially, 
                                                                   
		->double CalcYFromX(double x)
	Given a fitted matrix, this returns a Y-value for a specified X-value.
	
		->double ChiSqValue()
	Returns the Chi-Square value of the matrix, which is an indicator of
	the goodness of fit.
	
		->void FitData()
	Fits the current data set.
	
		->void GetCovarData(Covar &cov)
	Gets the covariance of the matrix
	
		->void GetFitParams(Params &params)
	Gets the parameters generated by the last fit
	
		->int NumParamsFitted()
	Returns the current order of polynomila fit
	
		->void SetNumParameters(int numParams)
	Sets the polynomial fit order
	
------------------------------------------------------------------- */

#pragma once

#include "WMFunctBase.h"

const int k_maxFitParams = 5;         // Highest fit order
const int k_maxFitData   = 100;       // Max # of data points in matrix

typedef double		Covar[k_maxFitParams + 1][k_maxFitParams + 1];
typedef double		Params[k_maxFitParams + 1];
typedef double		DataArr[k_maxFitData];
typedef double		DbyPArr[k_maxFitData][k_maxFitParams + 1]; 

// -------------------------------------------------------------------

class WMPolyFit : public WMFunctBase {
	public:
		WMPolyFit(DataFitSet* dArray, int maxPoints);
		
	virtual void
		BasisFunc(double x, Params &p);
	virtual double
		CalcYFromX(double x);
	virtual double
		ChiSqValue();
	virtual void
		FitData();
	void
		GetCovarData(Covar &cov);
	void
		GetFitParams(Params &params);
	int
		NumParamsFitted();
	void
		SetNumParameters(int numParams);
		
	bool
		ValidFit() {return mFitted;}
		
		
		private:
	double
		max(double a,double b) {return a > b ? a : b;}
	double
		sign(double a, double b);
	void
		SVBackSub(int dataPts, int numParams, DataArr &b);
	void
		SVDecomp(int dataPts, int numParams);
		
	DataFitSet*	mFitData;	
	Params		pArray;			// Array of calculated parameters
	DbyPArr 	u;				// Working arrays
	Covar		v;
	Params  	w;
	int 		mReqParams;		// # of requested parameters
	int 		mDataPts;		// # of datapoints collected from input
	int		mMaxArray;		// Highest index of data array
	bool		mFitted;		// Has data been fitted correctly?
	}; 