/* ------------------------------------------------------------------ 

	Title: WGStdGraph

	Description:  An class which makes a standard 2-D graph 
	skeleton using user-specified criteria and plots
	real data.

	Author :
		Stephen Wardlaw, M.D.
		Yale University School of Medicine
		20 York St.
		New Haven, CT  06504

	Edit History:
		07 Dec 97
			Adapted for the BeOS
		14 Feb 94	
			Rewritten again
		18 Jan 94
			Re-written for MetroWerks C++
		16 May 90
			Modified for MetroWerks Modula-2 on Macintosh
		06 Nov 87
			Last IBM Version 6.0
		?? Aug 80
			First version for Apple IIe
		
------------------------------------------------------------------ */ 
/*
	IMPLEMENTATION NOTES:
	
	This class will allow the production of very usable graphs with
	little additional code.
	
	The programmer may specify that the graph axes must be of equal 
	(graphic) length.In this case, the graph axes will be sized so that
	the longest one is shrunken to fit the other.
		
	The axes have a linear or log scale.  The client may also provide
	a transformation function.  If this is the case, the axes and any
	data subsequently plotted will be transformed accordingly.
*/		
//----------------------------------------------------------------------
//*  Notes on Methods		
//----------------------------------------------------------------------
/*
	->void ForceReDraw();
	Forces a complete re-drawing of the graph at the next update
	event, and creates an update event.
	
	->void PlotFunct(PlotFunction y_of_XFunct);
	Plots the value y := FunctBase::..CalcYFromX(x) from min to max X.

	->void PlotLine(double x1,y1,x2,y2);
	Plots a line between the two coordinates relative to the graph    
	scale in use.  No line is plotted if either end is out of the graph. 
	The color and line style are whatever is set.

	->bool PlotPoint(double X,Y);
	This procedure plots a point using the current marker characteristics.
	TRUE is returned if the point is inside of the graph bounds  

	->bool PlotTo(double X,Y);
	Plots from  the last positioned point to X,Y in the graph axes.   
	No line is plotted and FALSE is returned if the current point   
	is out of bounds. No line is plotted if the previous point was    
	out of bounds.

	->bool PosnToValue(Point pt,; double &x, double &y);
	Finds the real coordinate position from the graphic coordinate and   
	returns FALSE if the point is outside of the graph boundary.

	->bool PosnX(double x, int &X);
	Returns TRUE if "x" is within the current X-Axis limits and sets  
	"X" to the graph coordinate position.

	->bool PosnY(double y, int &Y);
	Returns TRUE if "y" is within the current Y-Axis limits and sets  
	"Y" to the graph coordinate position.
	
	->SetFrame();
	This sets up the internal graph coordinates so that the methods
	which plot the data will work properly.  This is normally called
	before the graph is drawn, but if you need to plot the data before
	then, call this first, after the drawing port is set.

	->double XformData(double val,
		                   bool onXAxis,
		                   bool toGraphUnits);
	This procedure transforms 'val' either to or from the actual 
	plotted graph units, for either the X ('xAxis' = TRUE) or Y axis.
	For example, if the X-Axis were a Log 10 scale:
		Xform(100.0,TRUE,TRUE) would return 2.0, and
		Xform(2.0,TRUE,FALSE) would return 100.0
	If a range error occurs, the function should not bomb but should return
	a very high number, a very low number, or zero, whichever is least
	likely to cause a major error.
*/
//----------------------------------------------------------------------

#pragma once

#include "WGAbscissaLabel.h"
#include "WGOrdinateLabel.h"
#include "WGAbscissa.h"
#include "WGOrdinate.h"
	

// Plot markers
enum EPlotMark{
	plotMark_none,				// Don't draw at all
	plotMark_spec,				// Special marking symbol 
	plotMark_dot,				// Very small square 'dot' 
	plotMark_plus,				// A plus sign 
	plotMark_smCirc,			// Small circle
	plotMark_diam,				// A diamond 
	plotMark_lgCirc,			// Large circle                                   
	plotMark_filledDot,		// Very small filled square 'dot' 
	plotMark_filledSmCirc,	// Small filled circle
	plotMark_filledLgCirc	// Large filled circle                                   
};

struct SPlotData{
	double x;
	double y;
	rgb_color	color;
	EPlotMark	mark;
};

// NOTE that this view will not automatically draw!
// The parent view is responsible for calling Draw().
		
class WGStdGraph : public BView, public BArchivable{
	
	public:
	
	// Create a graph from data contained in a BMessage
	static BArchivable*	
		Instantiate(BMessage* theArchive);
		
		WGStdGraph(BRect frame,  
					  const char* title	= "StdGraph",
					  uint32 mode			= B_FOLLOW_ALL_SIDES,
					  uint32 flags			= B_FULL_UPDATE_ON_RESIZE + 
												  B_FRAME_EVENTS); 

		WGStdGraph(BMessage* theMessage);
		
	virtual void
		Draw(BRect theRect);
	const char* const
		GetXLabel();
	const char* const
		GetYLabel();
	void
		ForceReDraw();
	virtual void
		PlotFunct(WMFunctBase* extFunct);
	virtual void
		PlotLine(double x1,double y1,double x2,double y2);
	virtual bool
		PlotPoint(double X,double Y);
	virtual bool
		PlotPoint(SPlotData* theData);
	virtual bool
		PlotTo(double X,double Y);
	virtual bool
		PosnToValue(BPoint pt, double &x, double &y);
	virtual bool
		PosnX(double x, float &X);
	virtual bool
		PosnY(double y, float &Y);
/*

	void		// Allow drag resizing
		SelectEnable(bool enable) {mSelectEnable = enable;}
	bool	// Selects a set of data points via the mouse
		SelectValue(double &xHi, double &xLo,
						double &yHi, double &yLo);
	---- */
	void
		SetAxesEqual(bool equal) {mEqualAxes = equal;}
	void		// Calculates the frame posn
		SetFrame();
	void
		SetGraphColor(rgb_color lineColor);
	void
		SetGraphFont(BFont* theFont);
	void
		SetPlotColor(rgb_color theColor) {mPlotColor = theColor;}
	void
		SetPlotMark(EPlotMark theMark) {mPlotMark = theMark;}
	void
		SetXLabel(const char* label);
	void
		SetYLabel(const char* label);
	double
		XformData(double val,
					 bool onXAxis,
					 bool toGraphUnits);
				  
	protected:	// Methods

	virtual void
		FrameMoved(BPoint thePt);
	virtual void
		FrameResized(float width, float height);
	void
		InitGraph();
		
	public:	// Data
	WGAbscissa*			mXAxis;			// The x-axis
	WGOrdinate*			mYAxis;			// The y-axis
   WGAbscissaLabel*	mXLabel;			// Label for x-axis
   WGOrdinateLabel*	mYLabel;			// Label for y-axis
  
  protected:	// Data
   
   enum {label_len = 32};
   
	BRect			mPlotArea;				// Display area for data (inside skeleton)
	BRect			mOldFrame;				// Last-drawn frame of graph
 	BPoint		mXLabOrigin;			// Where to draw X-Label
	BPoint		mYLabOrigin;			// Where to draw Y-Label
	float			mLastX;					// The last valid x-posn plotted 
	float			mLastY;					// The last valid y-posn plotted
	rgb_color	mPlotColor;				// Color of plot mark
	bool			mChangedAxes;			// Have the axes been changed?
	bool			mEqualAxes;				// Are X & Y axes equal?
	EPlotMark	mPlotMark;				// Type of plot mark to use
	//bool				mSelectEnable;	// Allow drag mouse cahnge of axes
	};
	