/* ------------------------------------------------------------------ 

	Title: WGStdGraph

	Description:  An class which makes a standard 2-D graph 
	skeleton using user-specified criteria and plots
	real data.

	Author :
		Stephen Wardlaw, M.D.
		Yale University School of Medicine
		20 York St.
		New Haven, CT  06504

	Edit History:
		07 Dec, 97
			Adapted for the BeOS
		14 Feb 94
			Rewritten again
		18 Jan 94
			Re-written for MetroWerks C++
		16 May 90
			Modified for MetroWerks Modula-2 on Macintosh
		06 Nov 87
			Last IBM Version 6.0
		?? Aug 80
			First version for Apple IIe
		
------------------------------------------------------------------ */ 

#include "WGStdGraph.h"
#include "WGAbscissa.h"
#include "WGOrdinate.h"

#include "UFloating_Util.h"
#include "UGraph_Util.h"

const float out_of_bounds = -1;

// -------------------------------------------------------------------
// *  Constructors & Destructor             
// -------------------------------------------------------------------
BArchivable*	
WGStdGraph::Instantiate(BMessage* theArchive)
{
   if (validate_instantiation(theArchive, "StdGraph") ) {
		return new WGStdGraph(theArchive); 
   } else {
   	return NULL;
   }
}

WGStdGraph::WGStdGraph(BMessage* theMessage):BView(theMessage)
{
	InitGraph();
}


WGStdGraph::WGStdGraph(BRect frame, 
							  const char* title,
							  uint32 mode, 
							  uint32 flags):BView(frame,title,mode,flags)
{
	InitGraph();
}
// -------------------------------------------------------------------
// *  Public Methods             
// -------------------------------------------------------------------
// Gets the pointer to the X-Axis label
const char* const
WGStdGraph::GetXLabel()
{
	return mXLabel->Text();
}
// -------------------------------------------------------------------
// Gets the pointer to the Y-Axis label
const char* const
WGStdGraph::GetYLabel()
{
	return mYLabel->Text();
}
// -------------------------------------------------------------------
// Forces a complete update of the graph
void
WGStdGraph::ForceReDraw() 
{
	mChangedAxes = true;
	Invalidate();
}
// -------------------------------------------------------------------
// Plots the value y := 'y_0f_XFunct(x)' from min to max X.
void
WGStdGraph::PlotFunct(WMFunctBase* extFunct) 
{
	double yMin,yMax,xMin,xMax,plotInt,val,exp1,y2;
	
	xMin = mXAxis->Min();
	yMin = extFunct->CalcYFromX(xMin);
	if(extFunct->MathError()) {
		yMin = mYAxis->Min();
	}
	if(mYAxis->Min() > yMin) {
		yMin = mYAxis->Min();
	}
	xMax = mXAxis->Max();
	yMax = extFunct->CalcYFromX(xMax);
	if(extFunct->MathError()) {
		yMax = mYAxis->Max();
	}
	if(mYAxis->Max() < yMax) {
		yMax = mYAxis->Max();
	}
	// Use transformed value to make 100 equal intervals.	
	plotInt = mXAxis->XformTo(xMax-xMin) / 100.0;
	val = mXAxis->XformTo(xMin);
	exp1 = xMin;
	mLastX = out_of_bounds;	// So won't plot first line
	while(exp1 <= xMax) {
		y2 = extFunct->CalcYFromX(exp1);
		if(!extFunct->MathError()) {
			PlotTo(exp1,y2);	// Plot if ok
		} else {
			mLastX = out_of_bounds;	// Set out-of-bounds
		}
		val += plotInt;
		exp1 = mXAxis->XformFrom(val);
	}
}	
// -------------------------------------------------------------------
// Plots a line between the two coordinates.
void
WGStdGraph::PlotLine(double x1,double y1,double x2,double y2) 
{
	float X1,Y1,X2,Y2;

	if(PosnX(x1,X1) && PosnX(x2,X2) &&
		PosnY(y1,Y1) && PosnY(y2,Y2) ) {
		MovePenTo(X1,Y1);
		BPoint thePt(X2,Y2);
		StrokeLine(thePt); 
		mLastX = X2;
		mLastY = Y2;
	}
}
// -------------------------------------------------------------------
// Plots a colored point.
bool
WGStdGraph::PlotPoint(SPlotData* theData) 
{	
	// Set color & mark
	bool valid = PlotPoint(theData->x,theData->y);
	// Reset color
	return valid;
}
// -------------------------------------------------------------------
// Plots a point.
bool
WGStdGraph::PlotPoint(double x,double y) 
{
	float X,Y;
	BRect r;
	BPoint pt1, pt2;
	
	if(mPlotMark == plotMark_none
		||	!mXAxis->ValToXPosn(x,X)
		||	!mYAxis->ValToYPosn(y,Y)) {
		
		mLastX = out_of_bounds;
		mLastY = out_of_bounds;
		return false;
	}
	switch (mPlotMark) {
		case plotMark_spec :
			r.Set(X,Y,X,Y);
			r.InsetBy(-3,-3);
			FillRect(r);
			break;
			
		case plotMark_plus :
			SetPenSize(1);
			BeginLineArray(2);
			pt1.Set(X,Y - 2);
			pt2.Set(X,Y + 2);
			AddLine(pt1,pt2,mPlotColor);
			pt1.Set(X - 2,Y);
			pt2.Set(X + 2,Y);
			AddLine(pt1,pt2,mPlotColor);
			EndLineArray();
			break;
			
		case plotMark_smCirc :
			r.Set(X,Y,X,Y);
			r.InsetBy(-2,-2);
			StrokeEllipse(r);
			break;
			
		case plotMark_diam :
			pt1.Set(X-2,Y);
			pt2 = pt1;
			pt2.x += 2;
			pt2.y -=2;
			BeginLineArray(4);
			AddLine(pt1,pt2,mPlotColor);
			pt1 = pt2;
			pt2.x += 2;
			pt2.y +=2;
			AddLine(pt1,pt2,mPlotColor);
			pt1 = pt2;
			pt2.x -= 2;
			pt2.y +=2;
			AddLine(pt1,pt2,mPlotColor);
			pt1 = pt2;
			pt2.x -= 2;
			pt2.y -=2;
			AddLine(pt1,pt2,mPlotColor);
			EndLineArray();
			break;
			
		case plotMark_lgCirc :
			r.Set(X,Y,X,Y);
			r.InsetBy(-4,-4);
			StrokeEllipse(r);
			break;
			
		case plotMark_filledDot :
			r.Set(X,Y,X,Y);
			r.InsetBy(-2,-2);
			FillRect(r);
			break;
			
		case plotMark_filledSmCirc :
			r.Set(X,Y,X,Y);
			r.InsetBy(-2,-2);
			FillEllipse(r);
			break;
			
		case plotMark_filledLgCirc :
			r.Set(X,Y,X,Y);
			r.InsetBy(-4,-4);
			FillEllipse(r);
			break;
			
		case plotMark_dot :
		default :
			r.Set(X,Y,X,Y);
			r.InsetBy(-2,-2);
			StrokeRect(r);
		}	
	mLastX = X;
	mLastY = Y;

	return true;
}
// -------------------------------------------------------------------
// Plots from  the last positioned point to X,Y in the graph axes.
bool
WGStdGraph::PlotTo(double x,double y) 
{
	float X,Y;

	if(!PosnX(x,X) || !PosnY(y,Y)) {
		mLastX = out_of_bounds;
		mLastY = out_of_bounds;
		return false;
	}
	if((mLastX == out_of_bounds) || (mLastY == out_of_bounds)) {
		mLastX = X;
		mLastY = Y;
		return true;
	}
	BPoint pt1(mLastX,mLastY);
	BPoint pt2(X,Y);
	StrokeLine(pt1,pt2);
	mLastX = X;
	mLastY = Y;
	
	return true;
}
// -------------------------------------------------------------------
// Finds the x & y data values from the graphic coordinate.
bool
WGStdGraph::PosnToValue(BPoint pt, double &x, double &y) 
{
	return mXAxis->XPosnToVal(pt.x,x) && mYAxis->YPosnToVal(pt.y,y);	
}
// -------------------------------------------------------------------
// Returns true if "X" is within the current X-Axis limits.
bool
WGStdGraph::PosnX(double x, float &X) 
{
	return mXAxis->ValToXPosn(x,X);	
}
// -------------------------------------------------------------------
// Returns true if "Y" is within the current Y-Axis limits.
bool
WGStdGraph::PosnY(double y, float &Y) 
{
	return mYAxis->ValToYPosn(y,Y);	
}
/*
// -------------------------------------------------------------------
// Selects a set of data points
bool
WGStdGraph::SelectValue(double &xHi, double &xLo,
								 double &yHi, double &yLo)
		
{
	// Set up a drawing mode for the selection box
	PenState normState, xState;
	Pattern pat;
	::GetPenState(&normState);	// Get normal pen state
	::GetIndPattern(&pat,0,2);
	::PenPat(&pat);
	::PenMode(patXor);
	::GetPenState(&xState);	//Get pen state for edit rectangle
	
	Point oldPt, origPt, pt;
	Rect fRect;
	bool ok = false;
	::GetMouse(&origPt); // Use this as starting point
	::SetPenState(&xState);
	oldPt = origPt;	// Keep track of point
	::SetRect(&fRect,origPt.h,origPt.v,origPt.h,origPt.v);
	int h1 = 0;
	int h2 = 0;
	int v1 = 0;
	int v2 = 0;	
	int hMin = mPlotArea.left;
	int hMax = mPlotArea.right;
	int vMin = mPlotArea.top;
	int vMax = mPlotArea.bottom;
	while(::StillDown()) {	// While mouse held down
		// Keep odometer going, if present
		::GetMouse(&pt);
		// Enforce limits on point
		if(pt.h > hMax) pt.h = hMax;
		if(pt.h < hMin) pt.h = hMin;
		if(pt.v > vMax) pt.v = vMax;
		if(pt.v < vMin) pt.v = vMin;
			
		if(!::EqualPt(pt,oldPt)) {	// If mouse has moved
			if(pt.h > origPt. h) {
				h1 = origPt.h;
				h2 = pt.h;
			} else {
				h1 = pt.h;
				h2 = origPt.h;
			}

			if(pt.v > origPt.v) {
				v1 = origPt.v;
				v2 = pt.v;
			} else {
				v1 = pt.v;
				v2 = origPt.v;
			}

		::FrameRect(&fRect);
		::SetRect(&fRect,h1,v1,h2,v2);
		::FrameRect(&fRect);
		oldPt = pt;
		}
	}
	
	::FrameRect(&fRect);
	::SetPenState(&normState);
	if(((h2 - h1) < 3) || ((v2 - v1) < 3)) {
		return false;	// EXIT HERE if too small
	}
	
	// The coords will be trimmed if out of bounds
	PosnToValue(origPt,xHi,yHi);
	PosnToValue(pt,xLo,yLo);
		
	double val;
	if(xLo > xHi) {	//Make sure they are in correct order
		val = xLo;
		xLo = xHi;
		xHi = val;
	}	
	if(yLo > yHi) {
		val = yLo;
		yLo = yHi;
		yHi = val;
	}	
	return true;
}
*/
// -------------------------------------------------------------------
void
WGStdGraph::SetGraphColor(rgb_color lineColor)
{
	mXAxis->SetHighColor(lineColor);			
	mXAxis->SetLowColor(ViewColor());	
			
	mYAxis->SetHighColor(lineColor);			
	mYAxis->SetLowColor(ViewColor());	

   mXLabel->SetHighColor(lineColor);			
	mXLabel->SetLowColor(ViewColor());	

 	mYLabel->SetHighColor(lineColor);	
	mYLabel->SetLowColor(ViewColor());	
	ForceReDraw();
}
// -------------------------------------------------------------------
void
WGStdGraph::SetGraphFont(BFont* theFont)
{
	if(Window() != NULL) {
		Window()->Lock();
	}
	SetFont(theFont);
	mXAxis->SetFont(theFont);			
	mYAxis->SetFont(theFont);			
   mXLabel->SetFont(theFont);	
   theFont->SetRotation(90);		
   mYLabel->SetFont(theFont);		
   ForceReDraw();
	if(Window() != NULL) {
		Window()->Unlock();
	}
}
// -------------------------------------------------------------------
// Calculates frame positioning as required
void
WGStdGraph::SetFrame()
{
	font_height		fHeight;
	BRect		gFrame;
	BRect		frame;
	BRect		xFrame;
	BRect		yFrame;
	BRect 	viewBounds;
	BPoint 	posn;
	BPoint	origin;
	float		yMargTop, yMargBot, yWd;
	float		xMargRt, xMargLt, xHt;
	float		xSize, ySize;
	float		xLabHt;
	float		yLabWd;
	float		frameHt; 
	float		frameWd;
	
	mChangedAxes = false;
	if(Window() != NULL) {
		Window()->Lock();
	}
	// Make room for the axis labels
	GetFontHeight(&fHeight);	// NOTE! All views have the same font
	xLabHt = (fHeight.ascent + fHeight.descent)*1.5;
	yLabWd = fHeight.ascent + fHeight.descent;
	
	// Calculate the frame below the title.
	mOldFrame = Frame();
	viewBounds = Bounds();
	
	// Now, fit the graph axis panels
	gFrame = Bounds();
	frameWd = gFrame.right - gFrame.left;
	frameHt = gFrame.bottom - gFrame.top;
	
	// Get size of XAxis labels
//	yLabWd = yFrame.right - yFrame.left;
	
	mXAxis->GetXMargins(xMargLt, xMargRt, xHt);
	mYAxis->GetYMargins(yMargTop, yMargBot, yWd);
	
	// Calc max size of axes and adjust if they are to be equal
	xSize = frameWd - (yWd + xMargLt + xMargRt + yLabWd);
	ySize = frameHt - (xHt + yMargTop +  yMargBot + xLabHt + xLabHt/3);
	if(mEqualAxes) {
		if(xSize > ySize) {
			xSize = ySize;
		} else {
			ySize = xSize;
		}
	}

	mYAxis->SizeAxisTo(ySize);
	mXAxis->SizeAxisTo(xSize);
	
	origin.x = yWd + yLabWd + yLabWd/3;;
	origin.y = frameHt - xHt - xLabHt + xLabHt/3;
	mXAxis->MoveXOriginTo(origin);
	mYAxis->MoveYOriginTo(origin);
	mPlotArea.Set(origin.x,origin.y - ySize, origin.x + xSize, origin.y);
	
	// Center the axes along the margins of the plot area
	xFrame = mXAxis->Frame();
	mXLabel->ResizeTo(mPlotArea.right - mPlotArea.left,Bounds().bottom - xFrame.bottom);
	mXLabel->MoveTo(mPlotArea.left,xFrame.bottom);
	
	yFrame = mYAxis->Frame();
	mYLabel->ResizeTo(yLabWd,mPlotArea.bottom - mPlotArea.top);
	mYLabel->MoveTo(0,mPlotArea.top);
	
	mLastX = out_of_bounds;
	mLastY = out_of_bounds;
	
	if(Window() != NULL) {
		Window()->Unlock();
	}
}
// -------------------------------------------------------------------
// Sets the X-Axis label
void
WGStdGraph::SetXLabel(const char* label) 
{
	mXLabel->SetText(label);
	mChangedAxes = true;
}
// -------------------------------------------------------------------
// Sets the Y-Axis label
void
WGStdGraph::SetYLabel(const char* label) 
{
	mYLabel->SetText(label);
	mChangedAxes = true;
}
// -------------------------------------------------------------------
// *  Protected Methods             
// -------------------------------------------------------------------
void
WGStdGraph::Draw(BRect theRect) 
{
	 if(mChangedAxes || mOldFrame != Frame()) {	// Always draw if resized
	 	SetFrame();
	 }
}
// -------------------------------------------------------------------
// NOTE! Until these work as advertized, we won't implement them
void
WGStdGraph::FrameMoved(BPoint thePt)
{
	// This could be called by the system or by the application.
	//mChangedAxes = true;
}		
// -------------------------------------------------------------------
// NOTE! Until these work as advertized, we won't implement them
void
WGStdGraph::FrameResized(float width, float height)
{
	// This could be called by the system or by the application.
	//mChangedAxes = true;
} 
// -------------------------------------------------------------------
// *  Private Methods             
// -------------------------------------------------------------------
// Initialize graph characteristics
void
WGStdGraph::InitGraph()
{
	BRect frame;
	
	// Add the X-Axis label
	frame.Set(0,0,0,0);
	mXLabel = new WGAbscissaLabel(frame);
	AddChild(mXLabel);

	// Add the Y-Axis label
	mYLabel = new WGOrdinateLabel(frame);
	AddChild(mYLabel);
	
	// Add the abscissa
	mXAxis = new WGAbscissa(frame);
	AddChild(mXAxis);
	
	// Add the ordinate
	mYAxis = new WGOrdinate(frame);
	AddChild(mYAxis);
	
	mPlotMark = plotMark_dot;
	mPlotColor = HighColor();
	mEqualAxes = false;
	mLastX = out_of_bounds;
	mLastY = out_of_bounds;
	mChangedAxes = true;
	//mSelectEnable = false;
	
}