/* ------------------------------------------------------------------ 

	Title: WGGraphAxis

	Description:  A class to be used in constructing graph axis.
		It provides base operations for the abscissa & ordinate.

	Author :
		Stephen Wardlaw, M.D.
		Yale University School of Medicine
		20 York St.
		New Haven, CT  06504

	Edit History:
		18 March, 1998
			Removed all structs to make it "endian-safe"
		09 Dec 97
			Adapted for BeOS
		09 Feb 94
			Split again from 'Abscissa' & 'Ordinate'
		24 Jan 94
			Split from the old Module Graph in MW M-2 and
			re-written for MetroWerks C++
		16 May 90
			Modified for MetroWerks Modula-2 on Macintosh
		06 Nov 87
			Last IBM Version 6.0
		?? Aug 80
			First version for Apple IIe
		
------------------------------------------------------------------ */ 

#pragma once

#include "WMFunctBase.h"
#include "URealFmt_Utils.h"

const float		border_wd		= 2;	// Width for drawing border & major tics
const double	min_num			= 1E-10;// Minimum allowable number
const double	rounding_err	= 1 + 1.0E-5;	// Label error
const float		tic_len			= 9;	// Length of a graph tic
const float 	tic_space		= 3;	// Space between tic and character

//The types of scales available
enum EAxisScale {
	scale_linear,
	scale_log2,
	scale_log10
	};
	
// Errors in constructing axes
enum EAxisError {
	axErr_noErr,
	axErr_span,
	axErr_logZero,
	axErr_badScale,
	axErr_interval
};
	
class WGGraphAxis : public WMFunctBase {

	public:

// Default constructor	
		WGGraphAxis();	
		
// Copy constructor		
		WGGraphAxis(const WGGraphAxis &theAxis);
		
	int16
		Divisions() {return mDivs;}
		
	double
		Interval() {return mInterval;}
		
	double
		Max() {return mMax;}
	
	double
		Min() {return mMin;}
		
	ERealFormat
		NumberFormat() {return mNumFmt;}
		
	int16
		NumberPrecision() {return mNumDPs;}
		
	EAxisScale
		Scale() {return mScale;}
			
	/*
		The following 'Set' methods try to set the axis parameters.
		If a value is out of range, or if there is a conflict amongst
		the settings, the method sets values to a non-destructive
		setting and returns the appropriate error message.
	*/
	EAxisError
		SetInterval(double interval);
		
	EAxisError
		SetLabelFormat(ERealFormat format, int16 numDPs);
		
	EAxisError
		SetMinorDivisions(int16 divs);
		
	EAxisError
		SetScale(EAxisScale newScale);
		
	EAxisError
		SetSpan(double min, double max);
		
	virtual double	// Axis->data transform
		XformFrom(double num);
		
	virtual double	// Data->axis transform
		XformTo(double num);
		
	protected:	// Methods
	
	void
		SetDefaults();
		
	EAxisError
		TestValues();
	
	protected:	// Data
	
	double			mMax;			// Axis limits
	double			mMin;	
	double			mInterval;	// Interval for labelling & major tics
	double			mLogBase;	// The base used for the log scale
	int16				mDivs;		// Minor axis divisions
	int16				mNumDPs;		// # of DPs/SigFigs
	EAxisScale		mScale;		// Scale type
	ERealFormat		mNumFmt;		// Axis marking format
	};
    
