#ifndef __PLUGIN__
#define __PLUGIN__

// (c)1997 P.Cisler
//
// Plugin API for Eddie
//
// This is an early version of the plugin API, make sure you
// check the distribution site for latest versions/more description
//
// you may also send email to pavel@be.com for more help with writing an
// Eddie plugin
//
// Current plugin API allows plugins to scan/modify/select text, scroll
// to different locations in the text, process mouse clicks in the text.
// The different API calls were added more or less to support the current
// suite of plugins that comes with Eddie. More calls will follow. The API
// is maintained in such a way as to prevent/decrease the likelyhood of breaking
// the compatibility with older plugins.
// The goal of the plugin API is to provide maximum extensibility and openness to
// the application, the plugin API will be evolved to meet that goal.
// If you are developing a plugin and are missing a certain call, let me know
// and I'll provide the call for you.
//
// By looking at the current call set you might find several unimplemented
// callbacks/selector calls. These are placeholders for functionality
// that will be added soon. Having these placeholders prevent potential breaking
// of existing plugins as the calls get implemented.
//
//
// The plugins are built as a shared library
//
// There are two entry points:
//
//	PluginInterface::PluginResult PluginMain(PluginInterface::PluginCallSelector, 
//		PluginInterface *);
//
//	PluginMain gets called periodically by Eddie with different selector calls. See
//	the following description of the different calls. Most selector calls are
//	called separately for each document window, the Window() getter will be set
//	to return the window the plugin is being called for.
//
//	const char *PluginHelp(int32 selector);
//
//	PluginHelp gets called to ask the plugin for text displayed in the tool tips; Just
//	have the plugin return the right text; selector is currently unused
//
//	You can use the following link line to build a plugin:
//
// 	mwld -o <yourPlugin> <yourPlugin>.o libbe.so libroot.so libdll.a libtracker.so -export pragma -xm s -map ResultingPlugin.xMAP
//
//	after building a plugin needs to get mimeset before it can be loaded by Eddie
//


#include <View.h>
#include <Window.h>
#include <Debug.h>
#include <stdlib.h>
#include <string.h>

#ifndef	_EXPORT
 #define _PLUGIN_EXPORT
 #define _PLUGIN_IMPORT
#else
 #if BUILDING_APP
  #define _PLUGIN_EXPORT _IMPORT
  #define _PLUGIN_IMPORT _EXPORT
 #else
  #define _PLUGIN_EXPORT _EXPORT
  #define _PLUGIN_IMPORT _IMPORT
 #endif
#endif


class PrefsPanel;
class _PLUGIN_IMPORT PluginInterface;

const int32 kEddieToPluginInterfaceVersion = 2;
	// this version indicates which set of the selectors the plugin fully supports; if the
	// existing selectors get renumbered or changed in some other incompatible way, this
	// number gets incremented and older plugins cease to work; Once the
	// API is stabilized, this is not likely to happen; Note that just adding a new selector
	// will typically not break existing plugins, the old plugin will just not be able to
	// take advantage of the call the new selector provides
const int32 kPluginToEddieInterfaceVersion = 10;
	// this version number would change if new calls/data members were added to
	// PluginInterface; in a case like this a new plugin, compiled with this new
	// PluginInterface layout would fail when ran on an older version of Eddie; further
	// the older version of Eddie might not provide all the selector calls the plugin
	// relies on

enum DocumentLanguageType {
	kUnknown,
	kShell,				// every file with no suffix
	kCC,
	kCH,
	kCCPlus,
	kCHPlus,
	kJava,
	kHTML,
	kPerl,
	kYacc,
	kLex,
	kAsm,
	kPascal,
	kModulaDef,
	kModulaOberonImpl,
	kFortran,
	kRez,
	kCLikeAsm,			// asm with C-style comments, ifdefs
	kMakefile			// makefile, Makefile, make*, etc.
};

// button part selector used for split buttons
// part press result codes
enum {
	kSolidButton = 0,
	kVerticallySplitButton = 1,
	kHorizontalySplitButton = 2,
	kFourWaySplitButton = 3,
	
	kEntireButton = 0,
	kTopButtonPart,
	kBottomButtonPart,
	kLeftButtonPart,
	kRightButtonPart,
	kLeftTopButtonPart,
	kRightTopButtonPart,
	kRightBottomButtonPart,
	kLeftBottomButtonPart
};

class PluginInterface {
public:

	enum PluginResult {
		kIgnored,		// did nothing during call
		kAccepted,		// handled call
		kExhausted,		// used with messages - fully handled message
		kFailed			// something screwed up
	};

	enum PluginCallSelector {
		kGetPluginVersion = 1, 	// never change this, otherwise version checking
								// will fail

		kInit,
			// called when Eddie starts up
		kUnload,
			// called when plugins are unloaded
		kQuit,
			// called when Eddie quit
		
		kButtonClick = 100,
			// plugin button clicked
		kButtonPress,
			// plugin button pressed
		kButtonDraw,
			// plugin button just draw, plugin may do some additional drawing to
			// the button
			// plugin must explicitly request this selector call

		kPrimitiveInvoked,
			// a keyboard shortcut associated with a primitive function handled
			// by this plugin was invoked
		kArgvPassed,
			// shell command handled by this plugin invoked; Eddie saves settings
			// as argv-style commands in the settings file; plugins may use the
			// same technique;
			// in future releases plugins will also be passed argv commands
			// they registered, executed from the shell

		kPulse = 200,
			// called during a pulse event of the text view
			// plugin must explicitly request this selector call
		kDraw,
			// called right after a draw call to the text view
			// plugin must explicitly request this selector call
		kMessage,
			// called from the beginning of the MessageReceived call of
			// the current text view
			// plugin must explicitly request this selector call
			// return kExhausted from plugin if the message is completely
			// processed in the call
		kMouseDown,
			// text view got a MouseDown event
			// plugin must explicitly request this selector call
		kKeyDown,
			// called from the beginning of the KeyDown call of
			// the current text view
			// plugin must explicitly request this selector call

		kWindowActivated = 300,
			// was kDocumentActivated
			// window brought to front
		kWindowDeactivated,
			// was kDocumentDectivated
			// document window sent to the background
		kDocumentTextChanged,
			// was kDocumentChanged
			// document text in window changed
		kDocumentOpened,
			// document opened - called for every document in multiple document
			// windows
		kDocumentClosed,
			// document closed
		kDocumentStateChanged,
			// document file renamed, saved, made readonly, writeable, etc.
		kDocumentActivated,
			// a document was activated; in single document windows this
			// is identical to kWindowActivated, if multiple documents in window
			// this gets called when one of the documents is made the current focus
		kWindowOpened,
			// window opened - time to install buttons
		kWindowClosed,
			// window closed
		
		kGlobalSettingsSaved = 400,
			// settings for the whole application are being saved, plugin
			// may save it's own settings using the WriteSettings call
		kDocumentSettingsSaved,
			// settings are being saved for the whole document, plugin may
			// save it's own settings as attributes in the documents node

		kGlobalPrefsShowing = 500,
			// the application settings window is being shown, plugin may
			// add it's own settings panel
		kDocumentPrefsShowing,
		
		// internal plugin calls
		kSyntaxColoringOff = 10000,
		kFindQuotedString
	};

	// some selector calls get sent only if a plugin requests them
	// because sending them to all plugins by default would be too
	// costly;
	// These selectors need to be explicitly requested by the plugin
	// by sending a mask of all the respective selector in the
	// RequireSelector during plugin initialization
	// 
	enum {
		kRequestButtonDrawMask = 0x1,
		kRequestPulseMask = 0x2,
		kRequestDrawMask = 0x4,
		kRequestMessageMask = 0x8,
		kRequestMouseDownMask = 0x10,
		kRequestKeyDownMask = 0x20,
		kRequestDocumentChangedMask = 0x40
	};
	
	PluginInterface();

	// version calls, used to determine if Eddie can use plugin and vice
	// versa	
	void ReturnSupportedPluginVersion(uint32 version
		= kEddieToPluginInterfaceVersion);
		// what the plugin guarantees to respond to
	void ReturnRequiredPluginVersion(uint32 version
		= kPluginToEddieInterfaceVersion);
		// what Eddie guarantees to respond to

	// the followng calls provide details for the different selector 
	// calls

	BWindow *Window() const;
		// the window plugin is currently being called from	
	uint32 DocumentUniqueID() const;
		// uinque ID of the document the plugin is being called on;
		// this is needed for windows with multiple documents -
		// the PluginMain is called for each document in multiple document
		// windows
	
	uint32 ClickSelector() const;
		// button part selector that got clicked/pressed

	// details about kDocumentTextChanged
	uint32 ChangeStart() const;
		// returns the starting position of a recent change
	uint32 RemovedLength() const;
		// returns the length of text removed by a recent change
	uint32 InsertedLength() const;
		// returns the length of text inserted by a recent change
	
	// details about kButtonDraw
	BRect ButtonRedrawRect() const;
	bool Inverted() const;
	
	// details about kKeyDown
	const char *KeyBytes() const;
	uint32 NumKeyBytes() const;
	
	// details about kMouseDown
	BPoint Point() const;
		// point of current mouse down

	// details about kMessage
	BMessage *CurrentMessage() const;
	
	// details about kDraw
	BRect CurrentDrawRect() const;
		
		
	virtual void SetupPluginButton(const unsigned char *buttonBits, 
		const unsigned char *pressedButtonBits);
		// show a button for this plugin with the supplied bits
		// call this during kWindowOpened
	virtual void ChangeButtonBitmaps(const unsigned char *buttonBits, 
		const unsigned char *pressedButtonBits, const BWindow *thisWindowOnly = 0);
		// call this to change the button bitmaps globally of in a specified window

	
	// text calls 
	virtual void GetSelection(int32 *start, int32 *end) const;
		// get the current selection
	virtual void Select(int32 start, int32 end);
		// change the current selection
	virtual int32 TextLength() const;
		// return the length of the current text

	// insert/remove calls; work on current selection, undo is handled for the
	// plugin, undo name may be specified
	virtual void Insert(const char *text, int32 length, const char *undoName = 0);
		// paste text into current selection; an undo name may be specified, otherwise
		// the undo menu item will use Plugin Paste
	virtual void Clear(const char *undoName = 0);
		// clear current selection; an undo name may be specified, otherwise
		// the undo menu item will use Plugin Remove	
	
	virtual void GetText(char *buffer, int32 offset, int32 length) const;
		// make a copy of <length> characters at <offset> and return the result
		// in buffer 
	virtual char CharAt(int32 offset) const;
		// get the character at offset

	virtual int32 CountLines() const;
		// return line count
	virtual int32 CurrentLine() const;
		// return the index of the line containing the selection start
	virtual void GoToLine(int32 lineNum);
		// select line

	virtual void ScrollToOffset(int32 inOffset);
		// scroll text to reveal offset
	virtual void ScrollToSelection();
		// scroll to reveal start of selection
	
	virtual void SetColor(rgb_color color);
		// sets a specified color on the current text selection
	
	// fast color setting calls for setting multiple ranges at one and forcing only
	// a single update
	virtual void StartSettingColorOnRange();
		// start setting
	virtual void SetOneColorOnRange(rgb_color color, int32 start, int32 end);
		// set one
	virtual void EndSettingColorOnRange();
		// done setting, force proper update

	virtual bool SyntaxColoring() const;
		// is syntax coloring on for this document?
	
	virtual PluginInterface &operator<<(const char *);
		// currently unused
	
	virtual void GetMouse(BPoint *, uint32 *, bool checkMessageQueue = true) const;
		// return mouse location and button states
	virtual int32 PointToOffset(BPoint) const;
		// conver a point in text to an offset
	
	virtual BRect TextViewBounds() const;
	virtual BRect WindowBounds() const;
	
	virtual const char *PluginName() const;
		// return the name of this plugin
	virtual BRect GetButtonRect() const;
		// window relative location of the plugins button
		
	
	virtual void ShowClipboardPanel(const char *text);
		// display a gray panel with supplied text under the plugin button;
		// keep showing while mouse is down 

	virtual void RegisterHandledShellCommand(const char *);
		// call during plugin initialization
		// used to publish argv style commands, handled by plugin
		// currently these are used to read settings during plugin initialization
	virtual void RegisterHandledPrimitiveCommand(const char *command,
		const char *description);
		// call during plugin initialization
		// used to publish primitive functions handled by plugin
		// primitives can be bound to a keyboard shortcut
	
	virtual void RequireSelectors(uint32 selectorMask);
		// call during plugin initialization
		// specify which of the explicit selectors are needed by the plugin
		// selectors that have to be explicitly required by a plugin are listed
		// with their respective masks in an enum earlier in this file
		
	virtual void AnimateButtonClick(int32 selector = 0);
		// call from when handling a keyboard primitive to simulate button
		// being cliked; selector currently unused; call this when your
		// plugin is responding to a primitive keyboard shortcut function
		// call that maps to a button click to give user a positive feedback


	// Information about current document
	virtual const BFile *DocumentNode() const;
		// the document is open as this BFile; returns null if new document
		// not yet saved; node changes after SaveAs
	virtual const entry_ref *DocumentRef() const;
		// the entry_ref of the document file; changes when document saved as,
		// renamed; the plugin gets notified about these changes be receiving the
		// selector
	virtual bool DocumentDirty() const;
		// returns true if document needs saving;
	virtual bool ReadOnly() const;
		// returns true if document cannot be modified on disk
	virtual bool IsShellWindow() const;
		// returns true if document is in shell mode
	virtual bool IsWorksheet() const;
		// returns true if document is the worksheet
	
	virtual DocumentLanguageType LanguageType() const;
		// returns the document language type

	virtual void WriteSettings(const char *format, ...);
		// called during global save settings call to save plugin settings
		// into the current settings file; plugin just writes it's settings string
		// using this call

	virtual void AddAppPrefsPanel(const char *, PrefsPanel *);
		// call to add a panel to the app settings window
	virtual void AddDocumentPrefsPanel(PrefsPanel *);
		// call to add a panel to the app settings window

	const char *CurrentPrimitive() const;
		// the keyboard primitive used to invoke plugin
	const char ***CurrentArgv() const;
		// reading settings or other argv command, registered by the plugin
		// was passed

	virtual rgb_color TextColor() const;
		// returns the TextColor of the current textView; syntax coloring
		// plugins use this to correctly revert coloring
	virtual rgb_color BackgroundColor() const;
		// returns the background color of the current text view

	// new plugin calls supporting split buttons
	virtual void SetupPluginButton(const unsigned char *buttonBits, 
		const unsigned char *pressedButtonBits, uint32 splitMode);
		// show a button for this plugin with the supplied bits
		// call this during kWindowOpened
	virtual void ChangeButtonBitmaps(const unsigned char *buttonBits, 
		const unsigned char *pressedButtonBits, uint32 splitMode,
		const BWindow *thisWindowOnly = 0);
		// call this to change the button bitmaps globally of in a specified window

	virtual int32 StartOfLine(int32 position) const;
		// returns the beginning of the line containing position
	virtual int32 EndOfLine(int32 position) const;
		// returns the end of the line containing position

	// insert/remove calls for more complex undo handling; work on current selection,
	// undo is initiated by StartUndo call; all calls to Insert and Clear are accumulated
	// within that undo
	virtual void StartUndo(int32 restoreSelStartTo, int32 restoreSelEndTo,
		const char *undoName = 0);
		// starts a new undo record with a specified or default (undo plugin) name
	virtual void SealUndo();
		// ends an undo record
	virtual void CurrentUndoInsert(const char *text, int32 length);
		// paste text into current selection; will be accumulated into the current undo record
	virtual void CurrentUndoClear();
		// clear current selection; will be accumulated into the current undo record

	BRect PrefsViewRect() const;
		// contains the rect for the preferece panel
	
	// special purpose calls
	uint32 Pos() const;
	uint32 *StartResult() const;
	uint32 *EndResult() const;

private:

	BWindow *targetWindow;
	void *textView;
	PluginCallSelector selector;	// copy of the selector passed in call main
	void *port;
	
	uint32 changeStart;
	uint32 removedLength;
	uint32 insertedLength;

	uint32 pos;
	uint32 *start;
	uint32 *end;

	BPoint point;
	BMessage *message;

	uint32 clickSelector;
	BRect redrawRect;

	uint32 requiredPluginVersion;
	uint32 supportedPluginVersion;

	bool inverted;

	const char *currentPrimitive;
	const char ***currentArgvCommand;	// currently unused
	
	const char *bytes;
	uint32 num;
	
	BRect rect;
	uint32 documentID;
	
friend class PluginList;
friend class PluginPort;
friend PluginPort *CallOnePlugin(PluginPort *, void *);
friend PluginPort *CheckOnePluginVersion(PluginPort *, void *);
};

class PrefsPanel {
public:
	PrefsPanel(BView *);
	virtual ~PrefsPanel();

	BView *View() const;

	virtual void Revert() = 0;
	virtual void Defaults() = 0;
	virtual void Apply() = 0;
	
	virtual bool CanRevert() const;
	virtual bool CanDefault() const;
	virtual bool CanApply() const;

	virtual bool ShowRevert() const;
	virtual bool ShowDefault() const;
	virtual bool ShowApply() const;

private:
	BView *panel;

friend class PrefsNameItem;
};

inline BWindow *
PluginInterface::Window() const
{
	return targetWindow;
}

inline uint32 
PluginInterface::ChangeStart() const
{
	ASSERT(selector == kDocumentTextChanged);
	return changeStart;
}

inline uint32 
PluginInterface::RemovedLength() const
{
	ASSERT(selector == kDocumentTextChanged);
	return removedLength;
}

inline uint32 
PluginInterface::InsertedLength() const
{
	ASSERT(selector == kDocumentTextChanged);
	return insertedLength;
}

inline BPoint 
PluginInterface::Point() const
{
	ASSERT(selector == kMouseDown);
	return point;
}

inline uint32 
PluginInterface::Pos() const
{
	ASSERT(selector == kFindQuotedString);
	return pos;
}

inline uint32 *
PluginInterface::StartResult() const
{
	ASSERT(selector == kFindQuotedString);
	return start;
}

inline uint32 *
PluginInterface::EndResult() const
{
	ASSERT(selector == kFindQuotedString);
	return end;
}

inline uint32 
PluginInterface::ClickSelector() const
{
	return clickSelector;
}

inline BRect 
PluginInterface::ButtonRedrawRect() const
{
	return redrawRect;
}

inline bool 
PluginInterface::Inverted() const
{
	return inverted;
}

inline void
PluginInterface::ReturnSupportedPluginVersion(uint32 version)
{
	supportedPluginVersion = version;
}

inline void
PluginInterface::ReturnRequiredPluginVersion(uint32 version)
{
	requiredPluginVersion = version;
}

inline const char *
PluginInterface::CurrentPrimitive() const
{
	ASSERT(selector == kPrimitiveInvoked);
	return currentPrimitive;
}

inline const char *** 
PluginInterface::CurrentArgv() const
{
	ASSERT(selector == kArgvPassed);
	return currentArgvCommand;
}

inline const char *
PluginInterface::KeyBytes() const
{
	ASSERT(selector == kKeyDown);
	return bytes;
}

inline uint32 
PluginInterface::NumKeyBytes() const
{
	ASSERT(selector == kKeyDown);
	return num;
}

inline BMessage *
PluginInterface::CurrentMessage() const
{
	ASSERT(selector == kMessage);
	return message;
}

inline BRect 
PluginInterface::CurrentDrawRect() const
{
	ASSERT(selector == kDraw);
	return rect;
}

inline BRect 
PluginInterface::PrefsViewRect() const
{
	ASSERT(selector == kGlobalPrefsShowing);
	return rect;
}

inline uint32 
PluginInterface::DocumentUniqueID() const
{
	return documentID;
}

extern "C" {

typedef PluginInterface::PluginResult (*PluginMainCall)(PluginInterface::PluginCallSelector, 
	PluginInterface *);
typedef const char *(*PluginHelpCall)(void);

#pragma export on

_PLUGIN_EXPORT
PluginInterface::PluginResult PluginMain(PluginInterface::PluginCallSelector, 
	PluginInterface *);

_PLUGIN_EXPORT
const char *PluginHelp(void);

#pragma export reset

}

#if 0

const char kExampleNormalBits = {
	0x1F,0x1F,0x1F,0x1F,0x1F,0x1F,0x1F,0x1F,0x1F,0x1F,0x1F,0x1F,0x1F,0x1F,0x1F,0x1C,
	0x1F,0x1E,0x1E,0x1E,0x1E,0x1E,0x16,0x00,0x00,0x16,0x1E,0x1E,0x1E,0x1E,0x1C,0x17,
	0x1F,0x1E,0x1E,0x1C,0x1C,0x1C,0x00,0x1E,0x1E,0x00,0x1C,0x1C,0x1C,0x1C,0x1A,0x17,
	0x1F,0x1E,0x15,0x00,0x00,0x00,0x00,0x0E,0x0E,0x00,0x00,0x00,0x00,0x15,0x1A,0x17,
	0x1F,0x1E,0x00,0x83,0x00,0x1E,0x1D,0x1E,0x1E,0x1E,0x1E,0x00,0x83,0x00,0x1A,0x17,
	0x1F,0x1E,0x00,0x83,0x00,0x1E,0x1E,0x1E,0x1E,0x1D,0x1E,0x00,0x83,0x00,0x1A,0x17,
	0x1F,0x1E,0x00,0x83,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x83,0x00,0x1A,0x17,
	0x1F,0x1E,0x00,0x83,0x83,0x83,0x83,0x83,0x83,0x83,0x83,0x83,0x83,0x00,0x1A,0x17,
	0x1F,0x1E,0x00,0x83,0x83,0x83,0x83,0x83,0x83,0x83,0x83,0x83,0x83,0x00,0x1A,0x17,
	0x1F,0x1E,0x00,0x83,0x83,0x83,0x83,0x83,0x83,0x83,0x83,0x83,0x83,0x00,0x1A,0x17,
	0x1F,0x1E,0x00,0x83,0x83,0x83,0x83,0x83,0x83,0x83,0x83,0x83,0x83,0x00,0x1A,0x17,
	0x1F,0x1E,0x00,0x83,0x83,0x83,0x83,0x83,0x83,0x83,0x83,0x83,0x83,0x00,0x1A,0x17,
	0x1F,0x1E,0x00,0x83,0x83,0x83,0x83,0x83,0x83,0x83,0x83,0x83,0x83,0x00,0x1A,0x17,
	0x1F,0x1E,0x00,0x83,0x83,0x83,0x83,0x83,0x83,0x83,0x83,0x83,0x83,0x00,0x1A,0x17,
	0x1F,0x1C,0x15,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x16,0x17,0x17,
	0x1C,0x17,0x17,0x17,0x17,0x17,0x17,0x17,0x17,0x17,0x17,0x17,0x17,0x17,0x17,0x15
};

const char kExamplePressedBits = {
	0x15,0x17,0x17,0x17,0x17,0x17,0x17,0x17,0x17,0x17,0x17,0x17,0x17,0x17,0x17,0x1C,
	0x17,0x17,0x1A,0x1A,0x1A,0x1A,0x16,0x00,0x00,0x15,0x1A,0x1A,0x1A,0x1A,0x1C,0x1F,
	0x17,0x1A,0x1B,0x1C,0x1C,0x1C,0x00,0x1E,0x1E,0x00,0x1C,0x1C,0x1C,0x1C,0x1E,0x1F,
	0x17,0x1A,0x16,0x00,0x00,0x00,0x00,0x0E,0x0E,0x00,0x00,0x00,0x00,0x16,0x1E,0x1F,
	0x17,0x1A,0x00,0x83,0x00,0x1E,0x1E,0x1E,0x1E,0x1E,0x1E,0x00,0x83,0x00,0x1E,0x1F,
	0x17,0x1A,0x00,0x83,0x00,0x1E,0x1E,0x1E,0x1E,0x1E,0x1E,0x00,0x83,0x00,0x1E,0x1F,
	0x17,0x1A,0x00,0x83,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x83,0x00,0x1E,0x1F,
	0x17,0x1A,0x00,0x83,0x83,0x83,0x83,0x83,0x83,0x83,0x83,0x83,0x83,0x00,0x1E,0x1F,
	0x17,0x1A,0x00,0x83,0x83,0x83,0x83,0x83,0x83,0x83,0x83,0x83,0x83,0x00,0x1E,0x1F,
	0x17,0x1A,0x00,0x83,0x83,0x83,0x83,0x83,0x83,0x83,0x83,0x83,0x83,0x00,0x1E,0x1F,
	0x17,0x1A,0x00,0x83,0x83,0x83,0x83,0x83,0x83,0x83,0x83,0x83,0x83,0x00,0x1E,0x1F,
	0x17,0x1A,0x00,0x83,0x83,0x83,0x83,0x83,0x83,0x83,0x83,0x83,0x83,0x00,0x1E,0x1F,
	0x17,0x1A,0x00,0x83,0x83,0x83,0x83,0x83,0x83,0x83,0x83,0x83,0x83,0x00,0x1E,0x1F,
	0x17,0x1A,0x00,0x83,0x83,0x83,0x83,0x83,0x83,0x83,0x83,0x83,0x83,0x00,0x1E,0x1F,
	0x17,0x1A,0x16,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x16,0x1E,0x1F,
	0x1C,0x1F,0x1F,0x1F,0x1F,0x1F,0x1F,0x1F,0x1F,0x1F,0x1F,0x1F,0x1F,0x1F,0x1F,0x1F
};

#endif

#if 1

inline
PrefsPanel::PrefsPanel(BView *panelView)
	:	panel(panelView)
{
}

inline
PrefsPanel::~PrefsPanel()
{
}

inline BView *
PrefsPanel::View() const
{
	 return panel;
}

inline bool 
PrefsPanel::CanRevert() const
{
	 return true;
}

inline bool 
PrefsPanel::CanDefault() const
{
	 return true;
}

inline bool 
PrefsPanel::CanApply() const
{
	 return true;
}

inline bool 
PrefsPanel::ShowRevert() const
{
	 return true;
}

inline bool 
PrefsPanel::ShowDefault() const
{
	 return true;
}

inline bool 
PrefsPanel::ShowApply() const
{
	 return true;
}

#endif

#endif