// VCard to Person File Converter
// Copyright 2002 Peter J. Goodeve
/***************************************************************
You are permitted to use this code for any purpose, provided that
the original author is acknowledged.
No warranty as to fitness for any particular purpose, nor responsibility
for any consequential damage, is expressed or implied.
****************************************************************/

#include <stdio.h>
#include "vcardparser.h"
#include <Application.h>
#include <Path.h>
#include <Entry.h>
#include <File.h>
#include <Directory.h>
#include <TypeConstants.h>

/*********************** People attributes to be handled:
----------  ---------   -------------------------------
  MIME str         21                         BEOS:TYPE
      Text          5                      META:company
      Text         18                      META:address
      Text          9                         META:city
      Text          3                        META:state
      Text          6                          META:zip
      Text          4                      META:country
      Text         15                       META:hphone
      Text          1                       META:wphone
      Text          1                          META:fax
      Text          1                        META:email
      Text          1                          META:url
      Text         13                        META:group (no)
      Text          4                     META:nickname (no)
      Text         14                         META:name
****************************/

#define PERSON_MIME_TYPE "application/x-person"

char* contactsdir = "/boot/home/people/Contacts/";


struct PersonFile {
	BString filename;
	VCard *vc;
	BFile f;
	PersonFile(): vc(NULL) {}
	PersonFile(VCard &vcard): vc(NULL) {createFile(vcard);}
	~PersonFile() {}
	bool createFile(VCard &vc);
	ssize_t writeAttr(char *attrname, BString &contents);
	ssize_t writeAttr(char *attrname, char *contents);
	bool addName();
	bool addAddress();	// inc citr,state,zip,country
	bool addHomePhone();
	bool addWorkPhone();
	bool addCompany();
	bool addEmail();
	bool addUrl();
	bool addFax();
//	bool addGroup();
//	bool addNickname();
	bool process(VCard &vcard);
};

bool PersonFile::createFile(VCard &vcard) {
	vc = &vcard;
	if (!vc->pick("N")) return false;	// can't do much here...;
	VCardItem *vi = vc->getPicked();
	if (!vi->hasText(0)) return false;	// just making sure...
	filename = "";
	if (vi->hasText(1)) {
	 	filename = *vi->text(1);	// first name
		filename += " ";
	}
	filename += *vi->text(0);	//surname
	filename.ReplaceAll('/', '+');	//no directory separators!
	while(filename.ByteAt(0) == ' ') filename.Remove(0,1);	// leading space is a nuisance
	while(filename.ByteAt(filename.Length()-1) == ' ')
		filename.Remove(filename.Length()-1,1);	// so is invisible trailing space...
	printf("making Person file %s:\n", filename.String());
	BString path(contactsdir);	// will have to ensure this exists
	path += filename.String();
	f.SetTo(path.String(), B_WRITE_ONLY | B_CREATE_FILE | B_ERASE_FILE);
	     // BeBook says that this erases all the attributes ^^^, but 'tain't so
	if (f.InitCheck() != B_OK) {
		printf("!!!--- COULDN'T CREATE FILE:\n   %s\n", path.String());
		return false;
	}
	f.Write("", 0);		// "touch" the file so date gets set
	f.WriteAttr("BEOS:TYPE", B_MIME_TYPE, 0,
				 PERSON_MIME_TYPE, sizeof(PERSON_MIME_TYPE));
	return true;
}

ssize_t PersonFile::writeAttr(char *attrname, BString &contents) {
	printf("   adding attr %s containing %s\n", attrname, contents.String());
	return f.WriteAttr(attrname, B_STRING_TYPE, 0,
			contents.String(), contents.Length()+1);
}

ssize_t PersonFile::writeAttr(char *attrname, char *contents) {
	BString s(contents);
	return writeAttr(attrname, s);
}


bool PersonFile::addName() {
	BString name;
	VCardItem *vi;
	if (vc->pick("FN")) {	// use Formatted/Full name if supplied
		vi = vc->getPicked();
		name = *vi->text();
	}
	else {
		vc->pick("N");
		vi = vc->getPicked();
		name = *vi->text(0);
		if (vi->hasText(1)) {
			name += ", ";
			name += *vi->text(1);
		}
		if (vi->hasText(2)) {	//possible middle name
			name += " ";
			name += *vi->text(2);
		}
		// Other segments might contain Title and Suffix -- ignored for now
//		if (vi->hasText(3)) {
//			name += " ";
//			name += *vi->text(3);
//		}
//		if (vi->hasText(4)) {
//			name += " ";
//			name += *vi->text(4);
//		}
	}
	writeAttr("META:name", name);
	return true;
}

bool PersonFile::addAddress() {
	int32 npicks = vc->pick("ADR");
	if (!npicks) return false;
	if (npicks > 1) npicks = vc->pick("ADR;HOME");
	if (!npicks) npicks = vc->pick("ADR;");	// give up and use the first...
	VCardItem *vi = vc->getPicked();
	BString streetaddr;
	if(vi->hasText(0)) {
		streetaddr = *vi->text(0);
		streetaddr += " ";
	}
	if(vi->hasText(1)) {
		streetaddr += *vi->text(1);
		streetaddr += " ";
	}
	if(vi->hasText(2)) 
		streetaddr += *vi->text(2);
	writeAttr("META:address", streetaddr);
	if(vi->hasText(3)) writeAttr("META:city", *vi->text(3));
	if(vi->hasText(4)) writeAttr("META:state", *vi->text(4));
	if(vi->hasText(5)) writeAttr("META:zip", *vi->text(5));
	if(vi->hasText(6)) writeAttr("META:country", *vi->text(6));
	return true;
}

bool PersonFile::addHomePhone() {
	VCardItem *vi;
	int32 npicks = vc->pick("TEL;HOME");
	if (npicks > 1) {
		vc->pick("TEL;HOME;CELL");
		vi = vc->getPicked();
		vi->processed();	// throw the mobile out...
		npicks = vc->pick("TEL;HOME");	// and try again
	}
	if (!npicks) npicks = vc->pick("TEL;");	// try the most basic...
	if (!npicks) return false;
	vi = vc->getPicked();
	if(vi->hasText()) writeAttr("META:hphone", *vi->text());
	return true;
}

bool PersonFile::addWorkPhone() {
	VCardItem *vi;
	int32 npicks = vc->pick("TEL;WORK");
	if (npicks > 1) {
		if (vc->pick("TEL;WORK;CELL")) {
	 		vi = vc->getPicked();
			vi->processed();	// throw the mobile out...
		}
		vc->pick("TEL;WORK;FAX");
		vi = vc->getPicked();
		if (vi) vi->processed();	// put the fax aside for a mo'
		npicks = vc->pick("TEL;WORK");	// and try again
		if (vi) vi->processed(false);	// put the fax back again now
	}
	if (!npicks) return false;
	vi = vc->getPicked();
	if(vi->hasText()) writeAttr("META:wphone", *vi->text());
	return true;
}

bool PersonFile::addCompany() {
	vc->pick("ORG");
	VCardItem *vi = vc->getPicked();
	if (!vi || !vi->hasText()) return false;
	writeAttr("META:company", *vi->text());
	return true;
}

bool PersonFile::addEmail() {
	VCardItem *vi;
	int32 npicks = vc->pick("EMAIL");
	vi = vc->getPicked();
	if (npicks > 1 && vc->pick("EMAIL;HOME")) {
		vi = vc->getPicked();
	}
	if (!vi || !vi->hasText()) return false;
	writeAttr("META:email", *vi->text());
	return true;
}

bool PersonFile::addUrl() {
	vc->pick("URL");
	VCardItem *vi = vc->getPicked();
	if (!vi || !vi->hasText()) return false;
	writeAttr("META:url", *vi->text());
	return true;
}

bool PersonFile::addFax() {
	vc->pick("TEL;FAX");	// accept any such...
	VCardItem *vi = vc->getPicked();
	if (!vi || !vi->hasText()) return false;
	writeAttr("META:fax", *vi->text());
	return true;
}

bool PersonFile::process(VCard &vcard) {
	if (!createFile(vcard)) return false;
	addName();
	addAddress();
	addHomePhone();
	addWorkPhone();
	addCompany();
	addEmail();
	addUrl();
	addFax();
	return true;
}


///////////////////////////////////////////////////////////

bool procVcardFile(FILE *f) {
	char buff[1024];
	VCard vcard;
	PersonFile pf;
	bool res, done=false, good=true;
	do {
		do {
			if (!fgets(buff, 1024, f)) {
				done = true;
				break;
			}
			res = vcard.addLine(buff);
		} while (!res);
		if (vcard.numLines() > 0 && !pf.process(vcard)) {
			printf("!!!--- Processing FAILED ---\n\n");
			good = false;
			vcard.clear();
			continue;
		}

		vcard.clear();
	} while (!done);
	return good;
}


class VPApp : public BApplication {
	bool hasargs;
public:
	VPApp() : BApplication("application/x-vcard2people"), hasargs(false) {
	}
	void ReadyToRun(void) {
		if (!hasargs) {
			if (!procVcardFile(stdin))
				printf("No VCard read from standard input\n");
		}
		PostMessage(B_QUIT_REQUESTED);
	}
	void ArgvReceived(int32 argc, char **argv) {
		hasargs = true;
		FILE *f;
		for (int i = 1; i < argc; i++) {
			f = fopen(argv[i],"r");
			if (!f) {
				printf("Couldn't find file %s\n", argv[1]);
				continue;
			}
			if (!procVcardFile(f))
				printf("\nVCard processing for %s had a failure!\n\n", argc >1? argv[1] : "stdin");
			fclose(f);
		}
	}	// end of ArgvReceived
	
	void RefsReceived(BMessage *msg) {
		uint32 enttype;
		int32 refcount;
		entry_ref argref;
		BEntry ent;
		BPath path;
		msg->GetInfo("refs", &enttype, &refcount);
		for (int i=0; i < refcount &&
			msg->FindRef("refs", i, &argref) == B_OK; i++) {
			if (ent.SetTo(&argref) == B_OK)
				ent.GetPath(&path);
				FILE *f = fopen(path.Path(),"r");
				if (f) procVcardFile(f);
				fclose(f);
		}
	}
};



int main(int argc, char **argv) {
	// Have to make sure this exists first!
	if (create_directory(contactsdir, 0777) != B_OK) {
		printf("couldn't create direectory %s!\n", contactsdir);
		return(11);
	}
	VPApp vpapp;
	vpapp.Run();
	return(0);
}