// Person File to VCard Converter
// Copyright 2002 Peter J. Goodeve
/***************************************************************
You are permitted to use this code for any purpose, provided that
the original author is acknowledged.
No warranty as to fitness for any particular purpose, nor responsibility
for any consequential damage, is expressed or implied.
****************************************************************/


#include <stdio.h>
#include <Application.h>
#include <Path.h>
#include <Entry.h>
#include <File.h>
#include <Directory.h>
#include <TypeConstants.h>
#include <String.h>

/*********************** People attributes to be handled:
----------  ---------   -------------------------------
  MIME str         21                         BEOS:TYPE
      Text          5                      META:company
      Text         18                      META:address
      Text          9                         META:city
      Text          3                        META:state
      Text          6                          META:zip
      Text          4                      META:country
      Text         15                       META:hphone
      Text          1                       META:wphone
      Text          1                          META:fax
      Text          1                        META:email
      Text          1                          META:url
      Text         13                        META:group (no)
      Text          4                     META:nickname (no)
      Text         14                         META:name
****************************/

#define PERSON_MIME_TYPE "application/x-person"

char* vcardfilename = "/boot/home/people/People.vcf";


struct PersonFile {
	BString filename;
	BFile f;
	FILE *vcardfile;
	PersonFile() : vcardfile(NULL) {}
	~PersonFile() {if (vcardfile) fclose(vcardfile);}
	bool createVCard(const char *filename);
	bool doneVCard();
	ssize_t readAttr(char *attrname, BString &contents);
	int32 extractWord(BString &src, BString &dst);
	bool readName();
	bool readAddress();	// inc citr,state,zip,country
	bool readHomePhone();
	bool readWorkPhone();
	bool readCompany();
	bool readEmail();
	bool readUrl();
	bool readFax();
	bool process(const char *filename);
};

bool PersonFile::createVCard(const char *filename) {
	if (!vcardfile) vcardfile = fopen(vcardfilename,"a");
	if (!vcardfile) return false;
	printf("opening Person file %s:\n", filename);
	f.SetTo(filename, B_READ_ONLY);
	if (f.InitCheck() != B_OK) return false;
	fprintf(vcardfile, "BEGIN:VCARD\r\n");
	return true;
}

bool PersonFile::doneVCard() {
	fprintf(vcardfile, "END:VCARD\r\n");
	return true;
}


ssize_t PersonFile::readAttr(char *attrname, BString &contents) {
	char buff[256] = "";
	ssize_t res = f.ReadAttr(attrname, B_STRING_TYPE, 0,
			buff, sizeof(buff));
	contents = buff;
	if (res > 1) printf("processing %s: %s\n", attrname, contents.String());
	return res > 0? res-1 : false;	// actual error not needed
}


int32 PersonFile::extractWord(BString &src, BString &dst) {	// src is modified
	char c;
	while (src.Length() && 
			((c = src.ByteAt(0)) == ' '
			|| c == ',' || c == ';'))
		src.Remove(0,1);	//lop off unwanted prepends
	int32 indx;
	for (indx = 0; indx < src.Length(); indx++) {
		c = src.ByteAt(indx);
		if (c == ' ' || c == ',' || c == ';') break;	// assume these are the relevant
	}
	if (indx) src.MoveInto(dst, 0, indx);
	else dst.SetTo("");
	return indx; 
}

bool PersonFile::readName() {
	BString name, vname, last, first, middle;
	readAttr("META:name", name);
	// This reies to be smart, but assumes no title or suffix!
	if (!extractWord(name, first)) return false;
	if (name.ByteAt(0) == ',') {	// assume Last is first...
		last = first;
		extractWord(name, first);
		extractWord(name, middle);	// if there
	}
	else {
		extractWord(name, last);
		if (name.Length()) {	// some left
			middle = last;	// must actually be the middle name
			extractWord(name, last);	// easy way of lopping separators
			last += name;	// and copy anything else too
		}
	}
	fprintf(vcardfile, "N:%s;%s;%s;;\r\n",
		last.String(), first.String(), middle.String());
	return true;
}

bool PersonFile::readAddress() {
	BString street, city, state, pcode, country;
	bool found = false;
	found |= readAttr("META:address", street);
	found |= readAttr("META:city", city);
	found |= readAttr("META:state", state);
	found |= readAttr("META:zip", pcode);
	found |= readAttr("META:country", country);
	if (found) {
		fprintf(vcardfile, "ADR:;;%s;%s;%s;%s;%s\r\n",
			street.String(), city.String(), state.String(),
			pcode.String(), country.String());
		return true;
	}
	return false;
}

bool PersonFile::readHomePhone() {
	BString phone;
	if (readAttr("META:hphone", phone))
		fprintf(vcardfile, "TEL;HOME:%s\r\n", phone.String());
	return true;
}

bool PersonFile::readWorkPhone() {
	BString phone;
	if (readAttr("META:wphone", phone))
		fprintf(vcardfile, "TEL;WORK:%s\r\n", phone.String());
	return true;
}

bool PersonFile::readCompany() {
	BString company;
	if (readAttr("META:company", company))
		fprintf(vcardfile, "ORG:%s\r\n", company.String());
	return true;
}

bool PersonFile::readEmail() {
	BString email;
	if (readAttr("META:email", email))
		fprintf(vcardfile, "EMAIL:%s\r\n", email.String());
	return true;
}

bool PersonFile::readUrl() {
	BString url;
	if (readAttr("META:url", url))
		fprintf(vcardfile, "URL:%s\r\n", url.String());
	return true;
}

bool PersonFile::readFax() {
	BString fax;
	if (readAttr("META:fax", fax))
		fprintf(vcardfile, "TEL;FAX:%s\r\n", fax.String());
	return true;
}

bool PersonFile::process(const char *filename) {
	if (!createVCard(filename)) return false;
	readName();
	readAddress();
	readHomePhone();
	readWorkPhone();
	readCompany();
	readEmail();
	readUrl();
	readFax();
	doneVCard();
	return true;
}


///////////////////////////////////////////////////////////

bool procPersonFile(const char *filename) {
	PersonFile pf;
	bool res, done=false;
	printf("processing %s\n", filename);
	pf.process(filename);
	return true;
}


class VPApp : public BApplication {
public:
	VPApp() : BApplication("application/x-people2vcard") {
	}
	void ReadyToRun(void) {
		PostMessage(B_QUIT_REQUESTED);
	}
	void ArgvReceived(int32 argc, char **argv) {
		for (int i = 1; i < argc; i++) {
			if (!procPersonFile(argv[i]))
				printf("VCard processing failed for %s\n", argc >1? argv[1] : "stdin");
		}
	}
	
	void RefsReceived(BMessage *msg) {
		uint32 enttype;
		int32 refcount;
		entry_ref argref;
		BEntry ent;
		BPath path;
		msg->GetInfo("refs", &enttype, &refcount);
		for (int i=0; i < refcount &&
			msg->FindRef("refs", i, &argref) == B_OK; i++) {
			if (ent.SetTo(&argref) == B_OK)
				ent.GetPath(&path);
				procPersonFile(path.Path());
		}
	}
};



int main(int argc, char **argv) {
	VPApp vpapp;
	vpapp.Run();
	return(0);
}