/**************************************************************************
 **
 ** Parser.cpp
 **
 ** A grammar for arithmetic expressions
 **
 ** +, -, *, /, cos, sin, tan, ^y, ^2, sqrt, (), -, integer, real
 **
 **************************************************************************
 **
 ** Author  : Nicolas Mougel
 ** E-Mail  : mougel@netcourrier.com
 **
 ** If you update this program, please contact the author.
 **
 **************************************************************************
 **
 ** The grammar used in the parser : 
 **
 ** 	TC  -> H S2 S
 **		S	-> +H S2 S | -H S2 S | #
 **		S2	-> *H S2 | / H S2 | #
 **		
 **		H	-> +H | -H | I
 **		I	-> cos H | sin H | tan H | acos H | asin H | atan H |
 **				sqrt H | exp H | ln H | J P
 **		J	-> ( TC ) | L
 **		P	-> ^H | #
 **		L	-> dM | d.C | bB | oO | xX | pi | M | .C | B | O | X 
 **
 **		M	-> D D1 N
 **		D	-> 0|1|2|3|4|5|6|7|8|9
 **		D1	-> D D1 | #
 **		N	-> .D1 | #
 **		C	-> DD1
 **
 **		B	-> B1 B2
 **		B1	-> 0|1
 **		B2	-> B1 B2 | #
 **
 **		0	-> O1 O2
 **		O1	-> 0|1|2|3|4|5|6|7
 **		O2	-> O1 O2 | #
 **
 **		X	-> X1 X2
 **		X1	-> 0|1|2|3|4|5|6|7|8|9|A|B|C|D|E|F
 **		X2	-> X1 X2 | #
 **
 **************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>

#include "Parser.h"
#include "Preferences.h"

#ifndef TRUE
#define TRUE 1
#endif
#ifndef FALSE
#define FALSE 0
#endif


/* Error
 */

#define N       6
#define OK      0
#define SYNTAX  1
#define DIV_Z   2
#define ROOT_N  3
#define LN_N	4
#define RANGE	5

/* The Preferences
 */

Preferences Prefs;

/* Fonctions' declarations
 */

int evaluate(char *result, char *string, int *pos, char **com);

int is_a_TC(double *x, char **p);
int is_a_S(double *x, char **p);
int is_a_S2(double *x, char **p);

int is_a_H(double *x, char **p);
int is_a_I(double *x, char **p);
int is_a_J(double *x, char **p);
int is_a_P(double *x, char **p);
int is_a_L(double *x, char **p);

int is_a_M(double *x, char **p);
int is_a_D(double *x, char **p);
int is_a_D1(double *x, char **p);
int is_a_N(double *x, char **p);
int is_a_C(double *x, char **p);

int is_a_B(double *x, char **p);
int is_a_B1(double *x, char **p);
int is_a_B2(double *x, char **p);

int is_a_O(double *x, char **p);
int is_a_O1(double *x, char **p);
int is_a_O2(double *x, char **p);

int is_a_X(double *x, char **p);
int is_a_X1(double *x, char **p);
int is_a_X2(double *x, char **p);

/* Utility 
 */

void deg_to_rad(double *angle);
void rad_to_deg(double *angle);
void dec_to_base(unsigned long a, int b, char chaine[]);
void reverse(char s[]);

int digit_number, err;

/**
 ** evaluate() :
 **
 **   Return the value of the arithmetic expression gived in the string
 **   Put the result in result
 **   Put the current character position in pos
 **	  The com variable is used to give the error message
 **
 **/

int evaluate(char *result, char *string, int *pos, char **com)
{
  double x;
  long int n;
  char *p;              /* pointer on the current character */
  char format[20], precision[20];

  char *lib[N]={"Good expression", 
                "Syntax Error", 
                "Division by zero",
                "Negative number in root",
                "Negative number in ln",
                "Out of range" };
  p=string;

  err=SYNTAX;
  *com=lib[err];

  /* precision of the parser
   */
  strcpy(format,"%.");
  sprintf(precision,"%d", Prefs.GetPrecision());
  strcat(format, precision);
  strcat(format, "lf");

  /* return true if string is null
   */
  if (string[0]=='\0')
  {
    return TRUE;
  }

  /* evaluate the expression
   */
  if (is_a_TC(&x,&p)==TRUE)
  {
  	if (Prefs.GetEuroMode()==P_EUR)
  		x=x*Prefs.GetEuro();
  	else if (Prefs.GetEuroMode()==P_CUR)
  		x=x/Prefs.GetEuro();
 
    sprintf(result,format,x);
  	if (Prefs.GetBaseOutput() != P_DEC)
  	{
      n=(long int) x;
      dec_to_base(n, Prefs.GetBaseOutput(), result);
	}
    if (*p=='\0')
    {
      return TRUE;
    }
    else
    {
      *pos=p-string;
      *com=lib[err];
      return FALSE;
    }
  }
  else
  {
    sprintf(result,format,x);
    *pos=p-string;
    *com=lib[err];
    return FALSE;
  }

}

/* deg_to_rad() :
 */

void deg_to_rad(double *angle)
{
	*angle=*angle*PI/180;
}

/* rad_to_def() :
 */

void rad_to_deg(double *angle)
{
	*angle=*angle*180/PI;
}

/* dec_to_base :
 *   convert a number a from base 10 to base b
 *   put the result in string
 */

void dec_to_base(unsigned long a, int b, char string[])
{
    int i = 0;
    char charmap[17] = "0123456789ABCDEF";

    do
    {
        string[i] = charmap[(a - (a/b) * b)];
        a = a / b;
        i++;
    }
    while (a != 0);
    string[i]='\0';

    reverse(string);
}

/* reverse a string
 */

void reverse(char s[])
{
    int i, j;
    char tmp;
    i = j = 0;
    while (s[j] != '\0')
        j++;
    j--;

    while(i < j)
    {
        tmp = s[i];
        s[i] = s[j];
        s[j] = tmp;
        i++;
        j--;
    }
}

/* is_a_TC() :
 */

int is_a_TC(double *x, char **p)
{
  if ( is_a_H(x,p)==TRUE && is_a_S2(x,p)==TRUE && is_a_S(x,p)==TRUE)
  {
    return TRUE;
  }
  return FALSE;
}

/* is_a_S() :
 */

int is_a_S(double *x, char **p)
{
  double y=0,z=0;
  while (**p==' ')
    *p=*p+1;
  if (**p=='+')
  {
    *p=*p+1;
    if ( is_a_H(&y,p)==TRUE && is_a_S2(&y,p)==TRUE && is_a_S(&z,p))
    {
      /* Add terms
       */
      *x=*x+y+z;
      return TRUE;
    }
    else
      return FALSE;
  }
  else if (**p=='-')
  {
    *p=*p+1;
    if ( is_a_H(&y,p)==TRUE && is_a_S2(&y,p)==TRUE && is_a_S(&z,p))
    {
      /* Substract terms
       */
      *x=*x-y+z;
      return TRUE;
    }
    else
      return FALSE;
  }

  return TRUE;
}

/* is_a_S2() :
 */

int is_a_S2(double *x, char **p)
{
  double y=1,z=1;
  while (**p==' ')
    *p=*p+1;
  if (**p=='*')
  {
    *p=*p+1;
    if ( is_a_H(&y,p)==TRUE && is_a_S2(&y,p)==TRUE)
    {
      /* Multiply terms
       */
      *x=*x*y;
      return TRUE;
    }
    else
      return FALSE;
  }
  else if (**p=='/')
  {
    *p=*p+1;
    if ( is_a_H(&y,p)==TRUE && is_a_S2(&z,p)==TRUE)
    {
      if (y==0 || z==0)
      {
        err=DIV_Z;
        return FALSE;
      }
      else
      {
        /* Divide terms
         */
        *x=(*x/y)*z;
        return TRUE;
      }
    }
    else
      return FALSE;
  }

  return TRUE;
}


/* is_a_H() :
 */

int is_a_H(double *x, char **p)
{
  while (**p==' ')
    *p=*p+1;
  if (**p=='+')
  {
    *p=*p+1;
    if ( is_a_H(x,p)==TRUE )
      return TRUE;
    else
      return FALSE;
  }
  else if (**p=='-')
  {
    *p=*p+1;
    if ( is_a_H(x,p)==TRUE )
    {
      *x=-*x;
      return TRUE;
    }
    else
      return FALSE;
  }
  else if ( is_a_I(x,p)==TRUE )
  {
  	return TRUE;
  }
  else
    return FALSE;
}

/* is_a_I() :
 */

int is_a_I(double *x, char **p)
{
  double y=0;

  while (**p==' ')
    *p=*p+1;
  if ( **p=='c' && *(*p+1)=='o' && *(*p+2)=='s' )
  {
    *p=*p+3;
    if ( is_a_H(&y,p)==TRUE )
    {
      if (Prefs.GetAngle()==P_DEG)
      	deg_to_rad(&y);
      *x=cos(y);
      return TRUE;
    }
    else
      return FALSE;
  }
  else if ( **p=='s' && *(*p+1)=='i' && *(*p+2)=='n' )
  {
    *p=*p+3;
    if ( is_a_H(&y,p)==TRUE )
    {
      if (Prefs.GetAngle()==P_DEG)
      	deg_to_rad(&y);
      *x=sin(y);
      return TRUE;
    }
    else
      return FALSE;
  }
  else if ( **p=='t' && *(*p+1)=='a' && *(*p+2)=='n' )
  {
    *p=*p+3;
    if ( is_a_H(&y,p)==TRUE )
    {
      if (Prefs.GetAngle()==P_DEG)
      	deg_to_rad(&y);
      *x=tan(y);
      return TRUE;
    }
    else
      return FALSE;
  }
  else if ( **p=='a' && *(*p+1)=='c' && *(*p+2)=='o' && *(*p+3)=='s')
  {
    *p=*p+4;
    if ( is_a_H(&y,p)==TRUE )
    {
      if ( y <-1.0 || y > 1.0)
      {
        err=RANGE;
        return FALSE;
      }
      else
      {
        *x=acos(y);
        if (Prefs.GetAngle()==P_DEG)
      	  rad_to_deg(x);
        return TRUE;
      }
    }
    else
      return FALSE;
  }
  else if ( **p=='a' && *(*p+1)=='s' && *(*p+2)=='i' && *(*p+3)=='n')
  {
    *p=*p+4;
    if ( is_a_H(&y,p)==TRUE )
    {
      if ( y <-1.0 || y > 1.0)
      {
        err=RANGE;
        return FALSE;
      }
      else
      {
        *x=asin(y);
        if (Prefs.GetAngle()==P_DEG)
      	  rad_to_deg(x);
        return TRUE;
      }
    }
    else
      return FALSE;
  }
  else if ( **p=='a' && *(*p+1)=='t' && *(*p+2)=='a' && *(*p+3)=='n')
  {
    *p=*p+4;
    if ( is_a_H(&y,p)==TRUE )
    {
      *x=atan(y);
      if (Prefs.GetAngle()==P_DEG)
      	rad_to_deg(x);
      return TRUE;
    }
    else
      return FALSE;
  }
  else if ( **p=='s' && *(*p+1)=='q' && *(*p+2)=='r' && *(*p+3)=='t')
  {
    *p=*p+4;
    if ( is_a_H(&y,p)==TRUE )
    {
      if (y<0)
      {
        err=ROOT_N;
        return FALSE;
      }
      else
      {
        *x=sqrt(y);
        return TRUE;
      }
    }
    else
      return FALSE;
  }
  else if ( **p=='l' && *(*p+1)=='n')
  {
    *p=*p+2;
    if ( is_a_H(&y,p)==TRUE )
    {
      if (y<=0)
      {
        err=LN_N;
        return FALSE;
      }
      else
      {
        *x=log(y);
        return TRUE;
      }
    }
    else
      return FALSE;
  }
  else if ( **p=='e' && *(*p+1)=='x' && *(*p+2)=='p')
  {
    *p=*p+3;
    if ( is_a_H(&y,p)==TRUE )
    {
      *x=exp(y);
      return TRUE;
    }
    else
      return FALSE;
  }
  else if ( is_a_J(x,p)==TRUE && is_a_P(x,p)==TRUE )
  {
    return TRUE;
  }
  else
    return FALSE;
}

/* is_a_J() :
 */

int is_a_J(double *x, char **p)
{
  while (**p==' ')
    *p=*p+1;
  if (**p=='(')
  {
    *p=*p+1;
    if ( is_a_TC(x,p)==TRUE )
    {
      if (**p==')')
      {
        *p=*p+1;
        return TRUE;
      }
      else
        return FALSE;
    }
    else
      return FALSE;
  }
  else if ( is_a_L(x,p)==TRUE )
  {
    return TRUE;
  }
  else
    return FALSE;
}

/* is_a_P() :
 */

int is_a_P(double *x, char **p)
{
  double y=1;
  while (**p==' ')
    *p=*p+1;
  if (**p=='^')
  {
    *p=*p+1;
    if ( is_a_H(&y,p)==TRUE )
    {
      *x=pow(*x,y);
      return TRUE;
    }
    else
      return FALSE;
  }

  return TRUE;
}

/* is_a_L() :
 */

int is_a_L(double *x, char **p)
{
  if (**p=='b')
  {
    *p=*p+1;
    if ( is_a_B(x,p)==TRUE )
    {
	  return TRUE;    
    }
    else
      return FALSE;
  }
  else if (**p=='o')
  {
    *p=*p+1;
    if ( is_a_O(x,p)==TRUE )
    {
	  return TRUE;    
    }
    else
      return FALSE;
  }
  else if (**p=='d')
  {
    *p=*p+1;
    if ( is_a_M(x,p)==TRUE )
    {
	  return TRUE;    
    }
    else if (**p=='.')
    {
      /* Compute decimal number
       */
      digit_number=0;
      *p=*p+1;
      if ( is_a_C(x,p)==TRUE )
      {
        while (digit_number>0)
        {
          *x=*x/10;
          digit_number--;
        }
	    return TRUE;    
      }
      else
        return FALSE;
    }
    else
      return FALSE;
  }
  else if (**p=='x')
  {
    *p=*p+1;
    if ( is_a_X(x,p)==TRUE )
    {
	  return TRUE;    
    }
    else
      return FALSE;
  }
  else if ( **p=='p' && *(*p+1)=='i' )
  {
  	*p=*p+2;
  	*x=PI;
  	return TRUE;
  }
  else if ( **p=='P' && *(*p+1)=='I' )
  {
  	*p=*p+2;
  	*x=PI;
  	return TRUE;
  }
  /* Use the defined base :
   * 	Not a "pure" grammar, so the code should be at the end
   * 	of the function
   */
  else if (Prefs.GetBase()==P_DEC)
  {
    if ( is_a_M(x,p)==TRUE )
    {
        return TRUE;
    }
    else if (**p=='.')
    {
      /* Compute decimal number
       */
      digit_number=0;
      *p=*p+1;
      if ( is_a_C(x,p)==TRUE )
      {
        while (digit_number>0)
        {
          *x=*x/10;
          digit_number--;
        }
	    return TRUE;    
      }
      else
      	return FALSE;
    }
    else
    	return FALSE;
  }
  else if (Prefs.GetBase()==P_BIN)
  {
    if ( is_a_B(x,p)==TRUE )
        return TRUE;
    else
    	return FALSE;
  }
  else if (Prefs.GetBase()==P_OCT)
  {
    if ( is_a_O(x,p)==TRUE )
        return TRUE;
    else
    	return FALSE;
  }
  else if (Prefs.GetBase()==P_HEX)
  {
    if ( is_a_X(x,p)==TRUE )
        return TRUE;
    else
    	return FALSE;
  }
  else
    return FALSE;
}

/* is_a_M() :
 */

int is_a_M(double *x, char **p)
{
  if ( is_a_D(x,p)==TRUE && is_a_D1(x,p)==TRUE && is_a_N(x,p)==TRUE )
  {
    return TRUE;
  }
  else
    return FALSE;
}

/* is_a_D() :
 */

int is_a_D(double *x, char **p)
{
  switch (**p)
  {
    case '0':
    case '1':
    case '2':
    case '3':
    case '4':
    case '5':
    case '6':
    case '7':
    case '8':
    case '9':
      digit_number++;
      *x=*(*p)-'0';
      *p=*p+1;
      return TRUE;
    default:
     return FALSE;
  }
}

/* is_a_D1() :
 */

int is_a_D1(double *x, char **p)
{
  double y=0;
  if ( is_a_D(&y,p)==TRUE )
  {
    /* Compute decimal number
     */
    *x=(*x*10)+y;
    if ( is_a_D1(x,p)==TRUE )
      return TRUE;
    else
      return FALSE;
  }
  else
    return TRUE;
}

/* is_a_N() :
 */

int is_a_N(double *x, char **p)
{
  double y;
  y=0.0;
  if (**p=='.')
  {
    /* Compute decimal number
     */
    digit_number=0;
    *p=*p+1;
    if ( is_a_D1(&y,p)==TRUE )
    {
      while (digit_number>0)
      {
        y=y/10;
        digit_number--;
      }
      *x=*x+y;
	  return TRUE;    
    }
    else
      return FALSE;
  }
  else
    return TRUE;
}

/* is_a_C() :
 */

int is_a_C(double *x, char **p)
{
  if ( is_a_D(x,p)==TRUE && is_a_D1(x,p)==TRUE )
    return TRUE;
  return FALSE;
}

/* is_a_B() :
 */

int is_a_B(double *x, char **p)
{
  if ( is_a_B1(x,p)==TRUE && is_a_B2(x,p)==TRUE )
    return TRUE;
  return FALSE;
}

/* is_a_B1() :
 */

int is_a_B1(double *x, char **p)
{
  switch (**p)
  {
    case '0':
    case '1':
      *x=*(*p)-'0';
      *p=*p+1;
      return TRUE;
    default:
     return FALSE;
  }
}

/* is_a_B2() :
 */

int is_a_B2(double *x, char **p)
{
  double y=0;
  if ( is_a_B1(&y,p)==TRUE )
  {
    /* Compute binary number
     */
    *x=(*x*2)+y;
    if ( is_a_B2(x,p)==TRUE )
      return TRUE;
    else
      return FALSE;
  }
  else
    return TRUE;
}

/* is_a_O() :
 */

int is_a_O(double *x, char **p)
{
  if ( is_a_O1(x,p)==TRUE && is_a_O2(x,p)==TRUE )
    return TRUE;
  return FALSE;
}

/* is_a_O1() :
 */

int is_a_O1(double *x, char **p)
{
  switch (**p)
  {
    case '0':
    case '1':
    case '2':
    case '3':
    case '4':
    case '5':
    case '6':
    case '7':
      *x=*(*p)-'0';
      *p=*p+1;
      return TRUE;
    default:
     return FALSE;
  }
}

/* is_a_O2() :
 */

int is_a_O2(double *x, char **p)
{
  double y=0;
  if ( is_a_O1(&y,p)==TRUE )
  {
    /* Compute octal number
     */
    *x=(*x*8)+y;
    if ( is_a_O2(x,p)==TRUE )
      return TRUE;
    else
      return FALSE;
  }
  else
    return TRUE;
}

/* is_a_X() :
 */

int is_a_X(double *x, char **p)
{
  if ( is_a_X1(x,p)==TRUE && is_a_X2(x,p)==TRUE )
    return TRUE;
  return FALSE;
}

/* is_a_X1() :
 */

int is_a_X1(double *x, char **p)
{
  switch (**p)
  {
    case '0':
    case '1':
    case '2':
    case '3':
    case '4':
    case '5':
    case '6':
    case '7':
    case '8':
    case '9':
      *x=*(*p)-'0';
      *p=*p+1;
      return TRUE;
    case 'A':
    case 'B':
    case 'C':
    case 'D':
    case 'E':
    case 'F':
      *x=10+*(*p)-'A';
      *p=*p+1;
      return TRUE;
    case 'a':
    case 'b':
    case 'c':
    case 'd':
    case 'e':
    case 'f':
      *x=10+*(*p)-'a';
      *p=*p+1;
      return TRUE;
    default:
     return FALSE;
  }
}

/* is_a_X2() :
 */

int is_a_X2(double *x, char **p)
{
  double y=0;
  if ( is_a_X1(&y,p)==TRUE )
  {
    /* Compute hexadecimal number
     */
    *x=(*x*16)+y;
    if ( is_a_X2(x,p)==TRUE )
      return TRUE;
    else
      return FALSE;
  }
  else
    return TRUE;
}

