// $Revision: 1.1 $
//	Copyright (C)1996 Be Inc.  All rights reserved.
// Modified by T. Yamagata for Calendar
//	Last Update: 981008

/*
	Author: William Adams
	Creation Date: 26 Dec 95
	
	Purpose:
	Implements the Gregorian View of a month
	Sunday = 0, Saturday = 6

	Sun Mon Tue Wed Thu Fri Sat
*/

#include <stdio.h>
#include <string.h>

#define	_BUILD_MONTH_VIEW_							//981006
#include "MonthView.h"
#include "GMCGraphics.h"

#include <Window.h>

//===================================================================
// Method: Constructor
//
// This is the typical constructor for a view.  The things
// of interest here are what flags are being set.
// FOLLOW_ALL - When the view is resized from any direction
//				grow the view in that direction.
// FULL_UPDATE_ON_RESIZE - Invalidate the entire view and not just the
//				freshly exposed portion.  We want this because otherwise
//				we would only redraw new portions, when in our case we
//				want the entire view to be drawn if the size changes.
// FRAME_EVENTS	We want to be notified when the view's size changes.
//				If we don't use this flag, then the FrameResized method
//				will not be called and we'll never know when to change
//				the size of the graphic.
//
// NOTES: If you want to see the effects of the various flags, try
//	removing them and see what happens.  Experimentation should give
//	a real clear understanding about how these things work.
//===================================================================
BTSMonthViewJ::BTSMonthViewJ(BRect frame, char *name, int32 m, int32 y) 
	: BView(frame, name, B_FOLLOW_ALL,
         B_FULL_UPDATE_ON_RESIZE|B_WILL_DRAW|B_FRAME_EVENTS)
{
	SetFont(be_bold_font);
	BFont font; GetFont(&font);
	font.GetFamilyAndStyle(&family, &style);
	InitMonthView(m, y, true);
}

//===================================================================
// Replicant Constructor
//===================================================================
BTSMonthViewJ::BTSMonthViewJ(BMessage *data)
	:BView(data)
{
	BFont font; GetFont(&font);
	strcpy((char *)family, data->FindString("family"));
	strcpy((char *)style, data->FindString("style"));
	font.SetFamilyAndStyle(family, style);
	SetFont(&font);
	
	InitMonthView(data->FindInt32("month"),
				data->FindInt32("year"),
				data->FindBool("showHolidays"));
	FrameResized(200., 180.);
}

//===================================================================
//   MonthView Initialization
//===================================================================
void
BTSMonthViewJ::InitMonthView(int32 m, int32 y, bool s)
{
	month = m;
	year = y;
	showHolidays = s;
	fBackgroundGraphic = new BTSMonthGraphic(this); 
	SetViewColor(B_TRANSPARENT_32_BIT);
}
	

//===================================================================
// Method: Destructor
//
// This is the average everyday variety of destructor.
// Here we simply cleanup after ourselves.
//===================================================================

BTSMonthViewJ::~BTSMonthViewJ()
{
	delete fBackgroundGraphic;
}


//===================================================================
//	Method: FrameResized
//
// Since we used the FRAME_EVENTS flag in the constructor
// of this view, this method will be called automatically
// whenever the view is resized.
//
// All we do here is set the size of the graphic.
// Invalidation and subsequent re-drawing of the view will
// happen automatically because we have used the FULL_UPDATE_ON_RESIZE
// flag in the constructor.
//===================================================================

void		
BTSMonthViewJ::FrameResized(float new_width, float new_height)
{
	fBackgroundGraphic->SetSize(BPoint(new_width, new_height));
}

//===================================================================
// Method: Draw
//
// This is called automatically by the view system whenever
// the view is newly exposed, or resized and things like that.
// This method simply calls the graphic, and tells it to draw.
// Since the graphic draws fairly quickly, there's really no need
// to create a backing store for the graphics to draw into, although
// this could be an optimization employed if the view becomes much
// more time consuming to draw.
//===================================================================

void
BTSMonthViewJ::Draw(BRect updateRect)
{
	fBackgroundGraphic->Draw(this);
}

//===================================================================
// Archive(): store data needed to replicate class
//===================================================================
extern const char *app_signature;
status_t
BTSMonthViewJ::Archive(BMessage *data, bool deep) const
{
	BView::Archive(data, deep);							//981008
	data->RemoveName("_resize_mode");
	data->AddInt32("_resize_mode", B_FOLLOW_NONE);
	data->AddString("class", "BTSMonthViewJ");
	data->AddString("add_on", app_signature);
	data->AddInt32("month", month);
	data->AddInt32("year", year);
	data->AddBool("showHolidays", showHolidays);
	data->AddString("family", family);
	data->AddString("style", style);
	return B_OK;
}

//===================================================================
// Instantiate(): Instantiate  the CalendarJ replicant
//===================================================================
BTSMonthViewJ
*BTSMonthViewJ::Instantiate(BMessage *data)
{
	if (validate_instantiation(data, "BTSMonthViewJ"))
		return new BTSMonthViewJ(data);
	// In case validation failed...
	return NULL;
}

void
BTSMonthViewJ::MessageReceived(BMessage *a_msg)
{
	switch (a_msg->what) {
	case  B_ABOUT_REQUESTED:
		AboutRequested();
		break;
	default:
		BView::MessageReceived(a_msg);					//981008
		break;
	}
}

void
BTSMonthViewJ::AboutRequested()
{
	BAlert  *alert;
	alert  = new BAlert("",
				"Replicant of CalendarJ v1.8",
				"OK");
	alert->Go();
}
