// $Revision: 1.1 $
//	Copyright (C)1995 Be Specific, Inc.  All rights reserved.
// Modified by T. Yamagata for Calendar 971126
//	Last update: 981008

/*
	Author: William Adams
	Creation Date: 20 Aug 93
	
	Purpose:
	Implements the Gregorian View of a month
	Sunday = 0, Saturday = 6

	Sun Mon Tue Wed Thu Fri Sat
*/

#include <stdio.h>
#include <string.h>
#include <time.h>

#include "GMCGraphics.h"
#include "StringLabel.h"
#include "MonthView.h"

#include <View.h>

//=======================================================

static char *GMCsingledaynames[] =
{
	"S",
	"M",
	"T",
	"W",
	"T",
	"F",
	"S"
};

static char *GMCshortdaynames[] =
{
	"日",
	"月",
	"火",
	"水",
	"木",
	"金",
	"土"
};

static char *GMCdaynames[] =
{
	"日曜日",
	"月曜日",
	"火曜日",
	"水曜日",
	"木曜日",
	"金曜日",
	"土曜日"
};

//static char *GMCshortmonthnames[] =
//{"",
//	"Jan",
//	"Feb",
//	"Mar",
//	"Apr",
//	"May",
//	"Jun",
//	"Jul",
//	"Aug",
//	"Sep",
//	"Oct",
//	"Nov",
//	"Dec"};
//
//static char *GMCmonthnames[] =
//{"",
//	"January",
//	"February",
//	"March",
//	"April",
//	"May",
//	"June",
//	"July",
//	"August",
//	"September",
//	"October",
//	"November",
//	"December"};

static int GMCmonthlengths[] = {31,28,31,30,31,30,31,31,30,31,30,31};

//static BTSStringLabel* GDayTextArray[32];

static int holidays[12][3] = {
	{ 0,14,-2}, {10,-2,-2}, {19,-2,-2},{28,-2,-2},
	{ 2, 3, 4}, {-2,-2,-2}, {19,-2,-2},{-2,-2,-2},
	{14,22,-2}, { 9,-2,-2}, { 2,22,-2},{22,-2,-2}};

static inline bool GMCisleap(int aYear)
{
    int newYear = aYear;
	bool leaping = FALSE;
	
    if (newYear < 100)
		newYear += 1900;

	leaping = (((((newYear)%4) == 0 && (((newYear)%100)!=0)) ||
		((newYear)%400)==0));
		
	return leaping;
}

static inline long GMCdaysInMonth(int m, int y)
{
    if ((m==2)&& GMCisleap(y))
        return 29;
    else
        return GMCmonthlengths[m-1];
}


//.......................................................................
BTSMonthGraphic::BTSMonthGraphic(BTSMonthViewJ *aView)
	: fView(aView),
	fdateLabel(0),
	fColumn0Label(0),
	fColumn1Label(0),
	fColumn2Label(0),
	fColumn3Label(0),
	fColumn4Label(0),
	fColumn5Label(0),
	fColumn6Label(0),
	fSizeIndex(0),
	fSize(100,100)
{		
	Init();
}


//........................................................................
BTSMonthGraphic::~BTSMonthGraphic()
{
}


//........................................................................
void
BTSMonthGraphic::Init()
{
	//fView->SetFont( be_bold_font );
	GridMargin = 4.0;
	GridTopLeft = BPoint(11.0, 43.0);
	GridCellSize = BPoint(34.0,28.0);
	GridBottomRight = BPoint(GridTopLeft.x+(7*GridCellSize.x),
				GridTopLeft.y+(6*GridCellSize.y));
	GridRect = BRect(GridTopLeft, GridBottomRight);	
	
    // Get the correct text to be displayed
	//long value = 1;
	
	//if (value > 12 || value < 0) 
	//	value = 1;
    
	// const char *monthname = GMCmonthnames[value];
    char datestring[64]={0};
	sprintf(datestring,"%ld 年 -  %ld 月", fView->year, fView->month);
	
	fdateLabel = new BTSStringLabel (datestring, BPoint(14,26),fMonthFontSize);

	// Create the day labels for each column
	fColumn0Label = CreateDayLabel(0);
	fColumn1Label = CreateDayLabel(1);
	fColumn2Label = CreateDayLabel(2);
	fColumn3Label = CreateDayLabel(3);
	fColumn4Label = CreateDayLabel(4);
	fColumn5Label = CreateDayLabel(5);
	fColumn6Label = CreateDayLabel(6);
	
	// Fill in the initial text array
	for (long counter=0; counter < 32; counter++)
	{
		char daystring[12]={0};
		sprintf(daystring,"%d",counter);
		
		GDayTextArray[counter] = new BTSStringLabel(daystring,BPoint(0,0),
			fNumberFontSize);
	}
	
	// Set the initial size
	BPoint boundsSize(fView->Bounds().Width(),fView->Bounds().Height());
	SetSize(boundsSize);
}

void BTSMonthGraphic::SetSize(BPoint size)
{
	fSize = size;
	Invalidate();
}

void BTSMonthGraphic::Invalidate()
{	
	// Do the resizing magic
	// Re-calculate the various size things
	CalculateParameters();
	CalculateDayLabels();
	
	// Make sure the view draws itself
	fImageNeedsUpdate = 1;
	
}


void BTSMonthGraphic::CalibrateFontSizes(const long sizeIndex)
{
	#ifdef DEBUG
	printf("CalibrateFontSizes - %d %d\n",fSizeIndex, sizeIndex);
	#endif
	
	// If the new index is the same as the old, then do nothing
	if (sizeIndex == fSizeIndex)
		return;
		
	switch (sizeIndex)
	{
		case 100:
			fNumberFontSize = 8.0;
			fMonthFontSize = 10.0;
		break;
		
		case 140:
			fNumberFontSize = 9.0;
			fMonthFontSize = 12.0;
		break;
		
		case 378:
			fNumberFontSize = 10.0;
			fMonthFontSize = 14.0;
		break;
		
		case 500:
			fNumberFontSize = 12.0;
			fMonthFontSize = 18.0;
		
		break;
		
		default:
			fNumberFontSize = 18.0;
			fMonthFontSize = 24.0;
		break;
	}
	
	#ifdef DEBUG
	printf("Calibrate - Resetting Text for numbers\n");
	#endif
	
	// Set the point size of the number displays
	for (long counter=0; counter < 32; counter++)
	{
		GDayTextArray[counter]->SetSize(fNumberFontSize);
	}
	
	fSizeIndex = sizeIndex;
}

void
BTSMonthGraphic::CalculateFontSizes()
{
	BRect bounds = fView->Bounds();
	long sizeIndex = 100;
	
	if (bounds.Width() < 100.0)
	{
		sizeIndex = 100;
	} else if (bounds.Width() < 140.0)
	{
		sizeIndex = 140;
	} else if (bounds.Width() < 378.0)
	{
		sizeIndex = 378;
	}else if (bounds.Width() < 500.0)
	{
		sizeIndex = 500;
	}else
	{
		sizeIndex = 1000;
	}
	
	CalibrateFontSizes(sizeIndex);
}

void
BTSMonthGraphic::CalculateParameters()
{
	// Do the calculation of the month stuff first
	// because everything else depends on it.
	CalculateFontSizes();

	// Re-Calculate the grid sizes and whatnot based on the clipped
	// size of the view.  Thus we can shrink the display to always
	// fit within the confines of the displayed area.
	BRect bounds = fView->Bounds();

	GridTopLeft = BPoint(GridMargin, 
					fMonthFontSize+fNumberFontSize+GridMargin*3);
	GridCellSize= BPoint((bounds.Width()-GridMargin*2)/7,
					((bounds.Height()-GridMargin*2-GridTopLeft.y)/6));
	GridBottomRight = BPoint(GridTopLeft.x+(7*GridCellSize.x),
					GridTopLeft.y+(6*GridCellSize.y));
	GridRect	= BRect(GridTopLeft, GridBottomRight);
}

void
BTSMonthGraphic::CalculateColumnLabel(BTSStringLabel &theLabel, int theColumn)
{
	// Calculate an origin such that the text will be centered
	BPoint center( GridTopLeft.x+theColumn*GridCellSize.x + (GridCellSize.x/2),
		 GridTopLeft.y-3.0);
	
	theLabel.SetText(fDayNameArray[theColumn]);
    theLabel.SetOrigin(center);
	theLabel.SetSize(fNumberFontSize);
	theLabel.SetAlignment(B_ALIGN_CENTER);
}

void
BTSMonthGraphic::CalculateDayLabels()
{
    //char **daynameArray;							//981008
	//char **monthnameArray;						//981008

	// determine the array to use for weekday names based on the width
	// of the columns.
	if (GridCellSize.x < 20.0)
	{
		fDayNameArray = GMCsingledaynames;
		//monthnameArray = GMCshortmonthnames;
	} else if (GridCellSize.x < 54.0)
	{
		fDayNameArray = GMCshortdaynames;
		//monthnameArray = GMCshortmonthnames;
	} else
	{
		fDayNameArray = GMCdaynames;
		//monthnameArray = GMCmonthnames;
	}

	// Create and assign a name for the month label
    //const char *monthname = GMCmonthnames[fView->month];
    char datestring[64]={0};
	//sprintf(datestring,"%s -  %ld",monthname, fView->year);
	sprintf(datestring,"%ld 年 -  %ld 月", fView->year,  fView->month);
	fdateLabel->SetText(datestring);
    fdateLabel->SetOrigin(BPoint (14.0,fMonthFontSize + GridMargin));
	fdateLabel->SetSize(fMonthFontSize);

	// Now Setup the day headings
	CalculateColumnLabel(*fColumn0Label, 0);
	CalculateColumnLabel(*fColumn1Label, 1);
	CalculateColumnLabel(*fColumn2Label, 2);
	CalculateColumnLabel(*fColumn3Label, 3);
	CalculateColumnLabel(*fColumn4Label, 4);
	CalculateColumnLabel(*fColumn5Label, 5);
	CalculateColumnLabel(*fColumn6Label, 6);
}


BTSStringLabel *
BTSMonthGraphic::CreateDayLabel(const int column)
{
	char **daynameArray;

	// determine the array to use for weekday names based on the width
	// of the columns.
	// If they are less than 58, then use the short names
	if (GridCellSize.x < 54.0)
	    daynameArray = GMCshortdaynames;
	else
		daynameArray = GMCdaynames;

	BTSStringLabel *newLabel = new BTSStringLabel(daynameArray[column]);
	//newLabel->SetColor(TTextColorStyle::GetBlack());
    newLabel->SetOrigin(BPoint (column*GridCellSize.x +GridTopLeft.x,40.0));
	newLabel->SetSize(fNumberFontSize);

	return newLabel;
}


static inline int Dow(int m, int y)
{
	int days[]={1,32,60,91,121,152,182,213,244,274,305,335};
	int k=y-1;
	
	long d=365*k+(k/4)-(k/100)+(k/400)+days[m-1];
	if (GMCisleap(y) && m>2) d++;
	return (int)d%7;
}


/*
//.........................................................................
	Method: Draw
 
	Purpose:  Draw the day numbers and any other shading specific to the days.

*/
void
BTSMonthGraphic::Draw(BTSMonthViewJ *aPort)
{
/*
	
	// Re-Calculate the grid sizes and whatnot based on the clipped
	// size of the view.  Thus we can shrink the display to always
	// fit within the confines of the displayed area.
*/
	BRect clipRect = fView->Bounds();
	
		
	if (fImageNeedsUpdate)
		Invalidate();
	CalculateDayLabels();

	// FIRST DRAW THE BACKGROUND
	// Draw a light gray background

//	aPort->SetHighColor(167,167,167);	
	// Draw a white background
	aPort->SetHighColor(255,255,255);
	aPort->FillRect(fView->Bounds());
	
	// Draw the outline that will hold the grid
	aPort->SetHighColor(255,255,255);	
	aPort->FillRect(GridRect);
	aPort->SetHighColor(0,0,0);	
	aPort->StrokeRect(GridRect);
		
    // Draw the text label for the month
	fdateLabel->Draw(aPort);
	
	// Draw the day labels
	fColumn0Label->Draw(aPort);
	fColumn1Label->Draw(aPort);
	fColumn2Label->Draw(aPort);
	fColumn3Label->Draw(aPort);
	fColumn4Label->Draw(aPort);
	fColumn5Label->Draw(aPort);
	fColumn6Label->Draw(aPort);
	
	// Draw the vertical lines in the grid.
	float xStart = GridTopLeft.x+1.0;
	float xEnd = GridBottomRight.x-1.0;
	float yStart = GridTopLeft.y+1.0;
	float yEnd = GridBottomRight.y-1.0;
	int counter = 0;

	
	// If the cell height is greater than 15.0 then draw the vertical
	// lines, otherwise they just get in the way.
	if (GridCellSize.y > 15.0)
	{
		rgb_color blackColor={0,0,0,0};
		fView->BeginLineArray(7);	
		for (counter = 1; counter < 7; counter++)
		{
			BPoint startPoint(xStart+(counter*GridCellSize.x)-2,yStart);
			BPoint endPoint(xStart+(counter*GridCellSize.x)-2,yEnd);
			//fView->StrokeLine(startPoint,endPoint);
			fView->AddLine(startPoint, endPoint, blackColor);
		}
		fView->EndLineArray();
	}
	
	// Draw the 5 horizontal lines in the grid.
	for (counter = 1; counter < 6; counter++)
	{	    
		BPoint startPoint(xStart,yStart+(counter*GridCellSize.y));
		BPoint endPoint(xEnd,yStart+(counter*GridCellSize.y));
		fView->StrokeLine(startPoint,endPoint);
	}

	// NOW DRAW EVERYTHING ELSE	
	// Get the day of the week of the first day in the month
	int mm = fView->month;
	int dow = Dow( mm, fView->year);
	
	// Get the number of days in the month
	int maxdays = GMCdaysInMonth( mm, fView->year);
	
    int currentRow = 0;
	int currentColumn = 0;

	// Draw the values of the current selection
    int dayctr = 0, holiday1, holiday2, holiday3;	//981008
	int tempDow = dow;
	bool showHolidays = false;
	
	// Holidays of the month
	if (fView->showHolidays) {
		showHolidays = true;
		holiday1 = holidays[mm-1][0];
		holiday2 = holidays[mm-1][1];
		holiday3 = holidays[mm-1][2];
	}
	
	// Here we go through each day of the month and find out what cell it should
	// be displayed in and put a number in the top left corner of the cell
	for (dayctr = 0; dayctr < maxdays; dayctr++)
	{
		float columnX = tempDow * GridCellSize.x + GridTopLeft.x + 2.0;
		float rowY = currentRow * GridCellSize.y + GridTopLeft.y + 1.0 + 
					fNumberFontSize;
		// Add a little extra for the font height
		
		// If the day is a weekday, droaw it in black.
		rgb_color dayColor = {0,0,0,0};
		
		// If it is Sunday, draw it in red.
		if (tempDow == 0) 
				dayColor.red  =  255;
		// If it's Saturday, draw it in blue.
		if (tempDow == 6) 
			dayColor.blue = 255;
		
		// If Japanese National Holidays are to be shown:	
		if (showHolidays) {
			if (dayctr==holiday1||dayctr==holiday2||dayctr==holiday3) {
				dayColor.red = 255;
				dayColor.blue = 0;
			}
			// If it is Monday, and the previous day is a National Holiday
			//    draw it in red.
			if (tempDow == 1)  {
				int	prevday = dayctr - 1;
				if (prevday==holiday1||prevday==holiday2||prevday==holiday3) 
					dayColor.red = 255;
			}
		}
				
		GDayTextArray[dayctr+1]->SetColor(dayColor);
		GDayTextArray[dayctr+1]->SetOrigin(BPoint(columnX,rowY));
		GDayTextArray[dayctr+1]->Draw(aPort);
		
	    // Increment the day of the week, if it is already  saturday == 6,
		// reset it to sunday == 0
	    if (tempDow == 6)
		{
			tempDow = 0;
			currentRow++;
		}else
		    tempDow++;
	}
	fImageNeedsUpdate = 0;

}

